
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.colors import LogNorm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Defining the colormap from white to green-blue
cmap = plt.cm.cubehelix

# Data for the two subplots (Philosophy metrics across different schools)
data1 = np.array(
    [
        [7.8, 55.0, 0.9, 45.0, 3.2, 76.1],
        [11.5, 35.2, 0.6, 40.0, 6.8, 52.3],
        [9.0, 80.5, 1.3, 78.2, 4.1, 65.0],
        [14.0, 25.0, 0.5, 30.2, 9.0, 40.0],
        [5.6, 90.3, 1.0, 85.0, 2.5, 55.3],
    ]
)

data2 = np.array(
    [
        [10.5, 70.0, 1.1, 90.0, 6.0, 65.1],
        [8.7, 45.0, 0.8, 55.0, 5.0, 85.2],
        [7.5, 65.5, 1.2, 68.0, 3.1, 70.0],
        [9.0, 35.0, 0.6, 50.0, 6.5, 60.0],
        [6.9, 85.0, 0.9, 80.0, 8.0, 58.5],
    ]
)

# X and Y labels
x_labels = ["Ethics", "Logic", "Metaphysics", "Epistemology", "Politics", "Aesthetics"]
y_labels = ["School A", "School B", "School C", "School D", "School E"]

# Subplot titles
titles = ["(a) Philosophy Metrics in Ancient Times", "(b) Philosophy Metrics in Modern Times"]

# Set up the colormap and norm (log scale)
norm = LogNorm(vmin=0.1, vmax=100)

# Axes Limits and Labels
xticks_values = range(len(x_labels))
yticks_values = range(len(y_labels))
colorbar_ticks = [0.1, 1, 10, 100]
yticklabels = ["0.1", "1", "10", "100"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set up the figure
fig, axes = plt.subplots(
    1, 2, figsize=(20, 8), gridspec_kw={"width_ratios": [1, 1], "wspace": 0.3}
)


# Function to create a subplot
def create_subplot(ax, data, title):
    # Create the scatter plot
    for i in range(data.shape[0]):
        for j in range(data.shape[1]):
            # Calculate the color based on the original value
            color = cmap(norm(data[i, j]))

            # Draw the circle with a fixed size
            circle = plt.Circle((j, i), 0.4, color=color)  # Fixed size
            ax.add_artist(circle)

            # Determine text color based on the value
            text_color = "white" if data[i, j] > 50 else "black"

            # Add the text inside the circle
            ax.text(
                j, i, f"{data[i, j]:.1f}", ha="center", va="center", color=text_color
            )

    # Set labels for x and y axes
    ax.set_xticks(range(len(x_labels)))
    ax.set_xticklabels(x_labels, ha="center")
    ax.set_yticks(range(len(y_labels)))
    ax.set_yticklabels(y_labels, va="center")

    # Adding the title for the subplot
    ax.set_title(title, fontsize=16)

    # Set the limits of the axes; they should be one more than your data range
    ax.set_xlim(-0.5, data.shape[1] - 0.5)
    ax.set_ylim(-0.5, data.shape[0] - 0.5)

    # Set the aspect of the plot to be equal and add a frame
    ax.set_aspect("equal")
    for spine in ax.spines.values():
        spine.set_visible(True)


# Create each subplot
create_subplot(axes[0], data1, titles[0])
create_subplot(axes[1], data2, titles[1])

# Create a colorbar on the far right side of the figure
sm = plt.cm.ScalarMappable(cmap=cmap, norm=norm)
sm.set_array([])
cbar = fig.colorbar(
    sm,
    ax=axes,
    ticks=colorbar_ticks,
    orientation="vertical",
    fraction=0.015,
    pad=0.05,
)
cbar.ax.set_yticklabels(yticklabels)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_85.pdf", bbox_inches="tight")

