
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.colors import LogNorm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Define the colormap from light green to dark green
cmap = plt.cm.Greens

# Sample data: a 5x6 grid, representing percentage distribution of students across subjects and grade levels
data = np.array(
    [
        [20.2, 18.0, 17.9, 15.0, 14.6, 14.3],
        [22.1, 21.3, 18.1, 13.5, 12.4, 12.6],
        [18.1, 23.6, 16.5, 14.8, 13.2, 13.8],
        [19.4, 17.8, 16.9, 14.4, 16.5, 15.0],
        [20.7, 21.4, 15.3, 14.7, 14.8, 13.1],
    ]
)

# X and Y labels
xticklabels = ["Math", "Science", "English", "History", "Art", "PE"]
yticklabels = ["Grade 1", "Grade 2", "Grade 3", "Grade 4", "Grade 5"]
x_label = "Subjects"
y_label = "Grade Levels"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set up the figure and axis
fig, ax = plt.subplots(figsize=(10, 8))

# Set up the colormap and norm (log scale)
norm = LogNorm(vmin=0.1, vmax=50)

# Create the scatter plot
for i in range(data.shape[0]):
    for j in range(data.shape[1]):
        # Calculate the color based on the original value
        color = cmap(norm(data[i, j]))

        # Draw the circle with a fixed size
        circle = plt.Circle((j, i), 0.4, color=color)  
        ax.add_artist(circle)

        # Determine text color based on the value
        text_color = "white" if data[i, j] > 20 else "black"

        # Add the text inside the circle
        ax.text(j, i, f"{data[i, j]:.1f}%", ha="center", va="center", color=text_color)

# Set labels for x and y axes
ax.set_xticks(range(len(xticklabels)))
ax.set_xticklabels(xticklabels, ha="center")
ax.set_yticks(range(len(yticklabels)))
ax.set_yticklabels(yticklabels, va="center")

# Adding titles for the axes
ax.set_xlabel(x_label, fontsize=14)
ax.set_ylabel(y_label, fontsize=14)

# Set the limits of the axes; they should be one more than your data range
ax.set_xlim(-0.5, data.shape[1] - 0.5)
ax.set_ylim(-0.5, data.shape[0] - 0.5)

# Set the aspect of the plot to be equal and add a frame
ax.set_aspect("equal")
for spine in ax.spines.values():
    spine.set_visible(True)

# Create a colorbar
sm = plt.cm.ScalarMappable(cmap=cmap, norm=norm)
sm.set_array([])
cbar = plt.colorbar(sm, ax=ax, ticks=[0.1, 1, 10, 50], orientation="vertical")
cbar.ax.set_yticklabels(["0.1", "1", "10", "50"])

# Add gridlines
plt.grid(True, which="both", color="gray", linestyle="--", linewidth=0.5)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout to fit the figure size
plt.tight_layout()

# Show the plot
plt.savefig("heatmap_91.pdf", bbox_inches="tight")
