
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.lines import Line2D  # Importing Line2D for creating custom legend items

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Data: Economics Context - GDP Growth and Unemployment Rate
years = [2015, 2016, 2017, 2018, 2019, 2020, 2021]
gdp_growth = [2.4, 1.8, 3.2, 2.9, 2.3, -1.8, 4.7]
unemployment_rate = [5.3, 5.5, 4.7, 4.4, 3.9, 8.1, 6.2]
avg_gdp_growth = [2.1] * len(years)
avg_unemployment_rate = [5.7] * len(years)

# Axes Limits and Labels
xlabel_value = "Years"
ylabel_value_1 = "GDP Growth (%)"
ylabel_value_2 = "Unemployment Rate (%)"
ylim_values_1 = [-3, 6]
ylim_values_2 = [3, 10]
yticks_values_1 = range(-3, 7, 1)
yticks_values_2 = range(3, 11, 1)

# Labels
label_gdp_growth = "GDP Growth"
label_unemployment_rate = "Unemployment Rate"
avg_label_gdp_growth = "Avg GDP Growth"
avg_label_unemployment_rate = "Avg Unemployment Rate"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plot
fig, ax1 = plt.subplots(figsize=(10, 6))

# Colors
color_gdp_growth = "#66c2a5"  # Soft green
color_unemployment_rate = "#fc8d62"  # Soft brown

# GDP Growth plot
(gdp_growth_line,) = ax1.plot(
    years,
    gdp_growth,
    "o-",
    color=color_gdp_growth,
    label=label_gdp_growth,
    markersize=10,
    linewidth=3,
    mec="black",
)
ax1.set_xlabel(xlabel_value, fontsize=14)
ax1.set_ylabel(ylabel_value_1, fontsize=14, color=color_gdp_growth)
ax1.tick_params(axis="y", labelcolor=color_gdp_growth, direction="in", rotation=90, labelsize=12)
ax1.tick_params(axis="x", direction="in", labelsize=12)
ax1.set_yticks(yticks_values_1)
ax1.set_ylim(ylim_values_1)

# Adding GDP Growth values to the plot
for i, txt in enumerate(gdp_growth):
    ax1.annotate(
        f"{txt}%",
        (years[i], txt),
        textcoords="offset points",
        xytext=(0, 10),
        ha="center",
        fontsize=12,
    )

# Unemployment Rate plot with a secondary y-axis
ax2 = ax1.twinx()
(unemployment_rate_line,) = ax2.plot(
    years,
    unemployment_rate,
    "s-",
    color=color_unemployment_rate,
    label=label_unemployment_rate,
    markersize=10,
    linewidth=3,
    mec="black",
)
ax2.set_ylabel(ylabel_value_2, color=color_unemployment_rate, fontsize=14)
ax2.tick_params(axis="y", labelcolor=color_unemployment_rate, direction="in", rotation=90, labelsize=12)
ax2.set_yticks(yticks_values_2)
ax2.set_ylim(ylim_values_2)

# Adding Unemployment Rate values to the plot
for i, txt in enumerate(unemployment_rate):
    ax2.annotate(
        f"{txt}%",
        (years[i], txt),
        textcoords="offset points",
        xytext=(0, -20),
        ha="center",
        fontsize=12,
    )

# Avg GDP Growth and Unemployment Rate dashed lines
ax1.axhline(y=2.1, color=color_gdp_growth, linestyle=":", linewidth=2)
ax2.axhline(y=5.7, color=color_unemployment_rate, linestyle="--", linewidth=2)

# Creating custom legend items
avg_gdp_growth_legend = Line2D(
    [0], [0], color=color_gdp_growth, linestyle=":", linewidth=2, label=avg_label_gdp_growth
)
avg_unemployment_rate_legend = Line2D(
    [0], [0], color=color_unemployment_rate, linestyle="--", linewidth=2, label=avg_label_unemployment_rate
)

# Adding legends
first_legend = ax1.legend(
    handles=[avg_gdp_growth_legend, avg_unemployment_rate_legend],
    loc="upper left",
    ncol=2,
    fontsize=14,
    edgecolor="black",
)
ax1.add_artist(first_legend)  # Add the first legend manually
second_legend = ax1.legend(
    handles=[gdp_growth_line, unemployment_rate_line],
    loc="upper right",
    fontsize=14,
    edgecolor="black"
)  # Add the second legend

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("line_172.pdf", bbox_inches="tight")
