

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(0)

# Updated steps
time = np.linspace(0, 365, 50)

# Generating distinct trends for each line
color_saturation = (np.sin(time * 0.017) * 20) + 50  # Sinusoidal trend for color saturation
brightness = np.array(time) ** 2 * 0.0002 + 30  # Quadratic growth for brightness
contrast = np.random.normal(loc=70, scale=10, size=len(time))  # Random noise for contrast
pattern_visibility = np.exp(-0.02 * time) * 80  # Exponential decay for pattern visibility

# Simulating standard deviations for error
std_dev_saturation = 5.0
std_dev_brightness = 7.0
std_dev_contrast = 10.0
std_dev_visibility = 8.0

# Data standard deviations
color_saturation_std = np.full_like(color_saturation, std_dev_saturation)
brightness_std = np.full_like(brightness, std_dev_brightness)
contrast_std = np.full_like(contrast, std_dev_contrast)
pattern_visibility_std = np.full_like(pattern_visibility, std_dev_visibility)

# Axes Limits and Labels
xlabel_value = "Time (days)"
xlim_values = [0, 365]
xticks_values = np.linspace(0, 365, 13)

ylabel_value_1 = "Color Saturation (%) / Brightness (%)"
ylabel_value_2 = "Contrast (%) / Pattern Visibility (%)"
yticks_values_1 = np.arange(0, 100, 10)
yticks_values_2 = np.arange(0, 110, 10)
ylim_values_1 = [0, 100]
ylim_values_2 = [-10, 110]

# Labels
label_1 = "Color Saturation (%)"
label_2 = "Brightness (%)"
label_3 = "Contrast (%)"
label_4 = "Pattern Visibility (%)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Creating a figure with two subplots
fig, (ax1, ax2) = plt.subplots(2, 1, figsize=(10, 12))

# Plotting on the first subplot
ax1.plot(time, color_saturation, "o-", color="#FFA500", label=label_1)
ax1.fill_between(time, color_saturation - color_saturation_std, color_saturation + color_saturation_std, color="#FFA500", alpha=0.2)

ax1.plot(time, brightness, "s-", color="#1E90FF", label=label_2)
ax1.fill_between(time, brightness - brightness_std, brightness + brightness_std, color="#1E90FF", alpha=0.2)

# Plotting on the second subplot
ax2.plot(time, contrast, "^--", color="#708090", markerfacecolor=(0, 0, 0, 0), markeredgecolor="#708090", label=label_3)
ax2.fill_between(time, contrast - contrast_std, contrast + contrast_std, color="#708090", alpha=0.2)

ax2.plot(time, pattern_visibility, "*-", color="#32CD32", label=label_4)
ax2.fill_between(time, pattern_visibility - pattern_visibility_std, pattern_visibility + pattern_visibility_std, color="#32CD32", alpha=0.2)

# Customizing the first subplot
ax1.set_xlabel(xlabel_value, fontsize=12)
ax1.set_ylabel(ylabel_value_1, fontsize=12)
ax1.set_xticks(xticks_values)
ax1.set_yticks(yticks_values_1)
ax1.set_xlim(xlim_values)
ax1.set_ylim(ylim_values_1)
ax1.legend(loc="upper center", frameon=False, ncol=2, bbox_to_anchor=(0.5, 1.15))
ax1.grid(True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5)
ax1.set_facecolor("#f9f9f9")

# Customizing the second subplot
ax2.set_xlabel(xlabel_value, fontsize=12)
ax2.set_ylabel(ylabel_value_2, fontsize=12)
ax2.set_xticks(xticks_values)
ax2.set_yticks(yticks_values_2)
ax2.set_xlim(xlim_values)
ax2.set_ylim(ylim_values_2)
ax2.legend(loc="upper center", frameon=False, ncol=2, bbox_to_anchor=(0.5, 1.15))
ax2.grid(True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5)
ax2.set_facecolor("#f9f9f9")

# Tight layout and saving output
plt.tight_layout()
plt.savefig("line_191.pdf", bbox_inches="tight")

