

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Updated steps
time_steps = np.linspace(0, 24, 50)

# Generating distinct trends for each line in entertainment domain
box_office_revenue = 50 + 2.5 * time_steps  # Linear growth
streaming_subscribers = np.log(time_steps + 1) * 10 + 20  # Logarithmic growth
daily_active_users = np.random.normal(loc=50, scale=10, size=len(time_steps))  # Random noise
movie_ratings = 100 * np.exp(-0.1 * time_steps)  # Exponential decay

# Simulating standard deviations for error
std_dev = 2.0
box_office_std = np.full_like(box_office_revenue, std_dev)
subscribers_std = np.full_like(streaming_subscribers, std_dev)
daily_active_users_std = np.full_like(daily_active_users, std_dev)
movie_ratings_std = np.full_like(movie_ratings, std_dev)

# Axes Limits and Labels
xlabel_value = "Time (months)"
xlim_values = [0, 24]
xticks_values = np.linspace(0, 24, 9)

ylabel_value_1 = "Revenue / Subscribers"
ylabel_value_2 = "Users / Ratings"
yticks_values_1 = np.arange(0, 130, 20)
yticks_values_2 = np.arange(0, 100, 10)
ylim_values_1 = [0, 120]
ylim_values_2 = [0, 100]

# Labels
label_box_office = "Box Office Revenue"
label_streaming_subscribers = "Streaming Subscribers"
label_daily_users = "Daily Active Users"
label_movie_ratings = "Movie Ratings"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Creating a figure with two subplots
fig, (ax1, ax2) = plt.subplots(2, 1, figsize=(6, 8))

# Plotting on the first subplot
ax1.plot(time_steps, box_office_revenue, "o-", color="#1f77b4", label=label_box_office)
ax1.fill_between(
    time_steps,
    box_office_revenue - box_office_std,
    box_office_revenue + box_office_std,
    color="#1f77b4",
    alpha=0.2,
)

ax1.plot(time_steps, streaming_subscribers, "s-", color="#ff7f0e", label=label_streaming_subscribers)
ax1.fill_between(
    time_steps,
    streaming_subscribers - subscribers_std,
    streaming_subscribers + subscribers_std,
    color="#ff7f0e",
    alpha=0.2,
)

# Plotting on the second subplot
ax2.plot(time_steps, daily_active_users, "^--", color="#2ca02c", label=label_daily_users)
ax2.fill_between(
    time_steps,
    daily_active_users - daily_active_users_std,
    daily_active_users + daily_active_users_std,
    color="#2ca02c",
    alpha=0.2,
)

ax2.plot(
    time_steps,
    movie_ratings,
    "*-",
    color="#d62728",
    label=label_movie_ratings,
)
ax2.fill_between(
    time_steps,
    movie_ratings - movie_ratings_std,
    movie_ratings + movie_ratings_std,
    color="#d62728",
    alpha=0.2,
)

# Customizing the first subplot
ax1.set_xlabel(xlabel_value, fontsize=12)
ax1.set_ylabel(ylabel_value_1, fontsize=12)
ax1.set_xticks(xticks_values)
ax1.set_yticks(yticks_values_1)
ax1.set_xlim(xlim_values)
ax1.set_ylim(ylim_values_1)
ax1.legend(loc="upper center", frameon=False, ncol=2, bbox_to_anchor=(0.5, 1.15))
ax1.grid(
    True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5
)
ax1.set_facecolor("#f9f9f9")

# Customizing the second subplot
ax2.set_xlabel(xlabel_value, fontsize=12)
ax2.set_ylabel(ylabel_value_2, fontsize=12)
ax2.set_xticks(xticks_values)
ax2.set_yticks(yticks_values_2)
ax2.set_xlim(xlim_values)
ax2.set_ylim(ylim_values_2)
ax2.legend(loc="upper center", frameon=False, ncol=2, bbox_to_anchor=(0.5, 1.15))
ax2.grid(
    True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5
)
ax2.set_facecolor("#f9f9f9")

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("line_192.pdf", bbox_inches="tight")

