
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)  # Seed for reproducibility
# Data for plotting in the context of seismic activity monitoring
time = np.array([1, 2, 3, 4, 5, 6])
seismic_activity = np.exp(-0.5 * time) + np.random.normal(0, 0.01, len(time))
ground_acceleration = 0.02 * np.cumsum(np.random.poisson(1, len(time)))
vibration_frequency = np.sin(time) * 0.1 + 0.1
noise_levels = np.random.uniform(0.01, 0.05, len(time))

# Axes Limits and Labels
xlabel_value = "Time (hours)"
ylabel_value = "Intensity"

# Labels
label_SeismicActivity = "Seismic Activity"
label_GroundAcceleration = "Ground Acceleration"
label_VibrationFrequency = "Vibration Frequency"
label_NoiseLevels = "Noise Levels"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
plt.figure(figsize=(10, 6))

# Plot the data using different styles and colors
plt.loglog(
    time,
    seismic_activity,
    "o-",
    color="teal",
    label=label_SeismicActivity,
    markerfacecolor="teal",
    markersize=8,
    linewidth=2,
)
plt.loglog(
    time,
    ground_acceleration,
    "x-",
    color="magenta",
    label=label_GroundAcceleration,
    markersize=8,
    linewidth=2,
)
plt.loglog(
    time,
    vibration_frequency,
    "s--",
    color="orange",
    label=label_VibrationFrequency,
    markersize=8,
    linewidth=2,
)
plt.loglog(
    time,
    noise_levels,
    "^-",
    color="navy",
    label=label_NoiseLevels,
    markersize=8,
    linewidth=2,
)

# Customize the x and y axes limits and labels
plt.xlabel(xlabel_value, fontsize=14)
plt.ylabel(ylabel_value, fontsize=14)
plt.xticks(time, labels=[str(t) + "h" for t in time])  # Ensuring x-ticks match the time values with unit
plt.xlim(
    left=time[0] * 0.9, right=time[-1] * 1.1
)  # Set x-axis limits to prevent cutting off data points
plt.ylim(
    bottom=1e-4, top=1e0
)  # Adjust y-axis limits based on new data range

# Adjust the legend position
plt.legend(bbox_to_anchor=(1, 0.5), fontsize=12, frameon=True)

# Add grid lines for better readability
plt.grid(True, which="both", ls="--", color="grey", linewidth=1, alpha=0.5)

# ===================
# Part 4: Saving Output
# ===================
# Adjust the layout and save the plot
plt.tight_layout()
plt.savefig("line_200.pdf", bbox_inches="tight")
