

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate distinct data sets for each subplot
N = np.array([1, 2, 3, 4, 5, 6])
datasets = {
    "Stock Prices": np.array([150, 160, 155, 170, 175, 180]) + np.random.uniform(low=-5, high=5, size=len(N)),
    "Trading Volume": np.array([500, 550, 600, 650, 700, 750]) + np.random.uniform(low=-40, high=40, size=len(N)),
    "Market Capitalization": np.array([2000, 2100, 2200, 2300, 2500, 2600]) + np.random.uniform(low=-100, high=100, size=len(N)),
    "Earnings per Share (EPS)": np.array([5, 5.5, 5.2, 5.8, 6.0, 6.2]),  # New dataset
}

# Assign each dataset to a subplot
plot_order = ["Stock Prices", "Trading Volume", "Market Capitalization", "Earnings per Share (EPS)"]

# Axes Limits and Labels
xlabel_value = "Quarters"
ylabel_value = "Financial Metrics"
ylim_values = [-100, 3000]

# Text
text = "Trend Analysis"
title = "Financial Metrics Study"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size and define colors, markers, and linestyles
fig, axs = plt.subplots(2, 2, figsize=(12, 8))  # Use a 2x2 subplot grid
colors = ["forestgreen", "royalblue", "darkorange", "chocolate"]
markers = ["o", "s", "^", "d"]
linestyles = ["-", "--", ":", "-."]

# Plot data in each subplot
for i, (ax, key) in enumerate(zip(axs.flat, plot_order)):
    for j, data_key in enumerate(plot_order):
        if key == data_key:
            ax.plot(
                N,
                datasets[data_key],
                linestyle=linestyles[j],
                marker=markers[j],
                color=colors[j],
                label=data_key,
                markersize=8,
            )
        else:
            ax.plot(
                N,
                datasets[data_key],
                linestyle=linestyles[j],
                marker=markers[j],
                color=colors[j],
                label=data_key,
                markersize=8,
                alpha=0.8,
            )  # Faded other lines

    ax.set_xlabel(xlabel_value)
    ax.set_ylabel(ylabel_value)
    ax.set_ylim(ylim_values)  # Ensure y-axis ranges don't clip data
    ax.set_title(f"{title} {i+1}")
    ax.legend()

    # Annotations to explain features, only on primary dataset for clarity
    if i == 0 or i == 3:
        ax.annotate(
            text,
            xy=(N[-2], datasets[key][-2]),
            xytext=(N[-4], datasets[key][-2] / 1.5 + 200),
            arrowprops=dict(arrowstyle="->", color="navy"),
            textcoords="data",
        )
    else:
        ax.annotate(
            text,
            xy=(N[-2], datasets[key][-2]),
            xytext=(N[-4], datasets[key][-2] * 1.1),
            arrowprops=dict(arrowstyle="->", color="navy"),
            textcoords="data",
        )

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout to prevent overlap and ensure clarity
plt.tight_layout()
plt.savefig("line_204.pdf", bbox_inches="tight")

