
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Generate new data representing tourism metrics over time
months = np.linspace(1, 12, 12)  # 12 months

# Simulate International and Domestic Visitors Trends
international_visitors = 100 + 30 * np.sin(2 * np.pi * (months - 3) / 12) + 20 * np.random.rand(12)
domestic_visitors = 50 + 10 * np.cos(2 * np.pi * (months - 1) / 12) + 10 * np.random.rand(12)

# Calculate uncertainty bounds for visual emphasis
upper_bound_international = international_visitors + 10
lower_bound_international = international_visitors - 10
upper_bound_domestic = domestic_visitors + 5
lower_bound_domestic = domestic_visitors - 5

# Axes Limits and Labels
xlabel_value = "Time (Months)"
xlim_values = [1, 12]
xticks_values = np.arange(1, 13, 1)

ylabel_value = "Number of Visitors (Thousands)"
ylim_values = [0, 150]
yticks_values = np.linspace(0, 150, 6)

# Labels
label_1 = "International Visitors"
label_2 = "Domestic Visitors"

# Title
title = "Tourism Trends Over Time"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size and create a single plot
fig, ax = plt.subplots(figsize=(10, 6))

# Define colors
color_international = "#006400"  # Dark green
color_domestic = "#696969"  # Dim grey
fill_color_international = "#98FB98"  # Light green
fill_color_domestic = "#D3D3D3"  # Light grey

# Plot international visitors with filled uncertainty
ax.fill_between(
    months,
    lower_bound_international,
    upper_bound_international,
    color=fill_color_international,
    alpha=0.3,
)
ax.plot(
    months,
    international_visitors,
    label=label_1,
    color=color_international,
    linestyle="-",
    linewidth=2,
    marker='o'
)

# Plot domestic visitors with filled uncertainty
ax.fill_between(
    months,
    lower_bound_domestic,
    upper_bound_domestic,
    color=fill_color_domestic,
    alpha=0.3,
)
ax.plot(
    months,
    domestic_visitors,
    label=label_2,
    color=color_domestic,
    linestyle="--",
    linewidth=2,
    marker='s'
)

# Customize the axes and grid
ax.set_title(title)
ax.set_xlabel(xlabel_value, fontsize=14)
ax.set_ylabel(ylabel_value, fontsize=14)
ax.set_xticks(xticks_values)
ax.set_yticks(yticks_values)
ax.grid(True, which='both', linestyle='--', linewidth=0.5)

# Add a legend to the plot
ax.legend(loc="upper right", frameon=True, fontsize=12)

# ===================
# Part 4: Saving Output
# ===================
# Enhance layout and display
plt.tight_layout()
plt.savefig("line_209.pdf", bbox_inches="tight")
