

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(1)

# Generate new data representing Crop Yields for two crops over time
time_points = np.linspace(0, 12, 100)  # Representing time in months
yield_crop_1 = 0.3 * np.sin(2 * np.pi * time_points / 12) + 0.7
yield_crop_2 = 0.2 * np.cos(2 * np.pi * time_points / 12) + 0.5

# Calculate uncertainty bounds for visual emphasis
upper_bound_crop_1 = yield_crop_1 + 0.1
lower_bound_crop_1 = yield_crop_1 - 0.1
upper_bound_crop_2 = yield_crop_2 + 0.1
lower_bound_crop_2 = yield_crop_2 - 0.1

# Axes Limits and Labels
xlabel_value = "Time (Months)"
xlim_values = [0, 13]
xticks_values = np.arange(0, 14, 1)

ylabel_value = "Crop Yield (tons per hectare)"
ylim_values = [0, 1.2]
yticks_values = np.linspace(0, 1.2, 6)

# Labels
label_1 = "Crop 1"
label_2 = "Crop 2"

# Title
title = "Monthly Crop Yields Over a Year"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size and create a single plot
fig, ax = plt.subplots(figsize=(10, 6))

# Color variables
fill_color_crop_1 = "#add8e6"
fill_color_crop_2 = "#ff9999"
line_color_crop_1 = "#1f77b4"
line_color_crop_2 = "#ff6347"

# Plot the crop yield for Crop 1 with filled uncertainty
ax.fill_between(
    time_points,
    lower_bound_crop_1,
    upper_bound_crop_1,
    color=fill_color_crop_1,
    alpha=0.3,
)
ax.plot(
    time_points,
    yield_crop_1,
    label=label_1,
    color=line_color_crop_1,
    linestyle="--",
    marker="o",
    linewidth=2,
)

# Plot the crop yield for Crop 2 with filled uncertainty
ax.fill_between(
    time_points,
    lower_bound_crop_2,
    upper_bound_crop_2,
    color=fill_color_crop_2,
    alpha=0.3,
)
ax.plot(
    time_points,
    yield_crop_2,
    label=label_2,
    color=line_color_crop_2,
    linestyle="-.",
    marker="x",
    linewidth=2,
)

# Customize the axes and grid
ax.set_title(title)
ax.set_xlabel(xlabel_value, fontsize=14)
ax.set_ylabel(ylabel_value, fontsize=14)
ax.set_xlim(xlim_values)
ax.set_ylim(ylim_values)
ax.set_xticks(xticks_values)
ax.set_yticks(yticks_values)
ax.grid(True, linestyle="--", alpha=0.7)

# Add a legend to the plot
ax.legend(loc="upper right", frameon=True, fontsize=12)

# ===================
# Part 4: Saving Output
# ===================
# Enhance layout and display
plt.tight_layout()
plt.savefig("line_210.pdf", bbox_inches="tight")
