
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(10)  # Different seed for new data generation

months = np.arange(1, 29)
version1_data = np.exp(-0.1 * months) + np.random.normal(loc=0, scale=0.05, size=len(months))
version1_data = np.clip(version1_data, 0, 1)
version2_data = 1 / (1 + np.exp(-0.1 * (months - 10))) + np.random.normal(loc=0, scale=0.05, size=len(months))
version2_data = np.clip(version2_data, 0, 1)
version3_data = 1 / (1 + np.exp(-0.1 * (months - 20))) + np.random.normal(loc=0, scale=0.05, size=len(months))
version3_data = np.clip(version3_data, 0, 1)

# Extracted variables
xlabel = "Months"
ylabel_version1 = "Version 1 Adoption Rate"
ylabel_version2 = "Version 2 Adoption Rate"
ylabel_version3 = "Version 3 Adoption Rate"
xlim = (0, 30)
ylim = (0, 1.2)
title = "Software Version Adoption Trends Over Time"
line_label_version1 = "Version 1"
line_label_version2 = "Version 2"
line_label_version3 = "Version 3"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis
fig, ax1 = plt.subplots(figsize=(10, 6))

# Plot the version 1 data on the primary y-axis
color = "#4682B4"  # SteelBlue
ax1.set_xlabel(xlabel)
ax1.set_ylabel(ylabel_version1, color=color)
(line1,) = ax1.plot(
    months,
    version1_data,
    color=color,
    marker="o",
    linestyle="-",
    clip_on=False,
    zorder=10,
    linewidth=2,
    label=line_label_version1,
)
ax1.tick_params(axis="y", labelcolor=color)
ax1.set_ylim(ylim)

# Create a secondary y-axis for the version 2 data
ax2 = ax1.twinx()
color = "#6A5ACD"  # SlateBlue
ax2.set_ylabel(ylabel_version2, color=color)
(line2,) = ax2.plot(
    months,
    version2_data,
    color=color,
    marker="s",
    linestyle="--",
    clip_on=False,
    zorder=10,
    linewidth=2,
    label=line_label_version2,
)
ax2.tick_params(axis="y", labelcolor=color)
ax2.set_ylim(ylim)

# Add a third line for version 3 data
color = "#32CD32"  # LimeGreen
(line3,) = ax1.plot(
    months,
    version3_data,
    color=color,
    marker="^",
    linestyle=":",
    linewidth=2,
    clip_on=False,
    zorder=10,
    label=line_label_version3,
)

# Customize the plot with a title, grid, and background color
fig.patch.set_facecolor("#f4f4f4")
ax1.set_facecolor("#e0f7fa")
ax2.set_facecolor("#fce4ec")
ax1.set_xlim(xlim)
ax1.tick_params(axis="both", which="both", length=0)
ax2.tick_params(axis="both", which="both", length=0)
lines = [line1, line2, line3]
labels = [line.get_label() for line in lines]
fig.legend(
    lines, labels, loc="upper center", bbox_to_anchor=(0.5, 0.95), ncol=3, frameon=False
)
# Set the title and display the plot
plt.title(title, y=1.05)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better spacing and display
plt.tight_layout()
plt.savefig("line_254.pdf", bbox_inches="tight")
