

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

months = list(range(1, 13))
product_sales = np.random.normal(loc=1000, scale=100, size=12)
ad_spend = np.random.normal(loc=50000, scale=5000, size=12)
customer_satisfaction = np.linspace(70, 90, 12) + (np.random.randn(12) * 2)

# Extracted variables
xlabel = "Months"
ylabel_sales = "Product Sales (units)"
ylabel_ad_spend = "Ad Spend ($)"
ylabel_satisfaction = "Customer Satisfaction (%)"
xlim = (0, 13)
ylim_sales = (500, 1500)
ylim_ad_spend = (30000, 70000)
ylim_satisfaction = (60, 100)
title = "Monthly Sales Metrics"
line_label_sales = "Product Sales"
line_label_ad_spend = "Ad Spend"
line_label_satisfaction = "Customer Satisfaction"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis
fig, ax1 = plt.subplots(figsize=(10, 6))

# Plot the product sales data on the primary y-axis
color_sales = "#2ca02c"  # Green color
ax1.set_xlabel(xlabel)
ax1.set_ylabel(ylabel_sales, color=color_sales)
(line1,) = ax1.plot(
    months,
    product_sales,
    color=color_sales,
    marker="o",
    linestyle="-",
    clip_on=False,
    zorder=10,
    linewidth=2,
    label=line_label_sales,
)
ax1.tick_params(axis="y", labelcolor=color_sales)
ax1.set_ylim(ylim_sales)

# Create a secondary y-axis for ad spend data
ax2 = ax1.twinx()
color_ad_spend = "#ff7f0e"  # Orange color
ax2.set_ylabel(ylabel_ad_spend, color=color_ad_spend)
(line2,) = ax2.plot(
    months,
    ad_spend,
    color=color_ad_spend,
    marker="x",
    linestyle="--",
    clip_on=False,
    zorder=10,
    linewidth=2,
    label=line_label_ad_spend,
)
ax2.tick_params(axis="y", labelcolor=color_ad_spend)
ax2.set_ylim(ylim_ad_spend)

# Adding customer satisfaction trend to the primary y-axis with secondary scale for clarity
ax3 = ax1.twinx()
# Offset the third y-axis
ax3.spines["right"].set_position(("outward", 60))
color_satisfaction = "#1f77b4"  # Blue color
ax3.set_ylabel(ylabel_satisfaction, color=color_satisfaction)
(line3,) = ax3.plot(
    months,
    customer_satisfaction,
    color=color_satisfaction,
    marker="^",
    linestyle=":",
    linewidth=2,
    clip_on=False,
    zorder=10,
    label=line_label_satisfaction,
)
ax3.tick_params(axis="y", labelcolor=color_satisfaction)
ax3.set_ylim(ylim_satisfaction)

# Add a legend to the plot
lines = [line1, line2, line3]
labels = [line.get_label() for line in lines]
fig.legend(
    lines, labels, loc="upper center", bbox_to_anchor=(0.5, 0.95), ncol=3, frameon=False
)

# Customize the plot with a title, grid, and background color
fig.patch.set_facecolor("#f4f4f4")
ax1.set_facecolor("#e5f5f9")
ax2.set_facecolor("#f9e5e6")
ax3.set_facecolor("#eaf5e6")
ax1.set_xlim(xlim)
ax1.grid(True, which='both', linestyle='--', linewidth=0.5)
ax1.tick_params(axis="both", which="both", length=0)
ax2.tick_params(axis="both", which="both", length=0)
ax3.tick_params(axis="both", which="both", length=0)
plt.title(title, y=1.05)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better spacing and display
plt.tight_layout()
plt.savefig("line_255.pdf", bbox_inches="tight")

