
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Data for the plot
data_sizes = np.array([100, 200, 300, 400, 500, 600, 700])
algorithm_a_speed = np.clip(
    (1000 / data_sizes) + np.random.normal(0, 0.5, len(data_sizes)),
    1,
    10,
)
algorithm_b_speed = np.clip(
    (1100 / (data_sizes + 50)) + np.random.normal(0, 0.5, len(data_sizes)),
    1,
    10,
)
highlight = [200, 400, 600]

# Axes Limits and Labels
xlabel_value = "Data Size (MB)"
ylabel_value_1 = "Processing Speed (A) [Ops/sec]"
ylabel_value_2 = "Processing Speed (B) [Ops/sec]"

# Labels
label_1 = "Algorithm A"
label_2 = "Performance Threshold"

label = "Algorithm B"

# Main title and subplot titles
main_title = "Processing Speed of Algorithms A and B Across Data Sizes"
subplot_title_1 = "Algorithm A Processing Speed"
subplot_title_2 = "Algorithm B Processing Speed"

# Annotation text
annotation_text = "Critical Point"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with a 1x2 grid
fig, axs = plt.subplots(1, 2, figsize=(14, 6))

fig.suptitle(main_title, fontsize=16)

# Plot Algorithm A data
(ax_a,) = axs[0].plot(
    data_sizes,
    algorithm_a_speed,
    "o-",
    color="#1f77b4",
    label=label_1,
    markersize=8,
    linewidth=2,
)
axs[0].set_xlabel(xlabel_value, fontsize=12)
axs[0].set_ylabel(ylabel_value_1, fontsize=12)
axs[0].tick_params(axis="y", direction="in", labelsize=10)
axs[0].tick_params(axis="x", direction="in", labelsize=10)
axs[0].set_title(subplot_title_1, fontsize=14)

# Plot Algorithm B data
(ax_b,) = axs[1].plot(
    data_sizes,
    algorithm_b_speed,
    "s--",
    color="green",
    label=label,
    markersize=8,
    linewidth=2,
)
axs[1].set_xlabel(xlabel_value, fontsize=12)
axs[1].set_ylabel(ylabel_value_2, fontsize=12)
axs[1].tick_params(axis="y", direction="in", labelsize=10)
axs[1].tick_params(axis="x", direction="in", labelsize=10)
axs[1].set_title(subplot_title_2, fontsize=14)

# Add a legend to the plot
threshold = 3.5
axs[0].axhline(y=threshold, color="red", linestyle="-", linewidth=1.5, label=label_2)
axs[1].axhline(y=threshold, color="red", linestyle="-", linewidth=1.5, label=label_2)

# Highlight the data points above the threshold
for ax in [axs[0], axs[1]]:
    for size in highlight:
        ax.plot(
            size, algorithm_a_speed[np.where(data_sizes == size)], "ro"
        ) if ax == axs[0] else ax.plot(
            size, algorithm_b_speed[np.where(data_sizes == size)], "ro"
        )
        ax.annotate(
            annotation_text,
            (size, algorithm_a_speed[np.where(data_sizes == size)])
            if ax == axs[0]
            else (size, algorithm_b_speed[np.where(data_sizes == size)]),
            textcoords="offset points",
            xytext=(0, -20),
            ha="center",
        )

# Customize the plot with labels, title, and legend
axs[0].legend()
axs[1].legend()

# Add a grid to the plot
for ax in axs:
    ax.grid(True, linestyle="--", alpha=0.6)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better spacing and display
plt.tight_layout()
plt.savefig("line_280.pdf", bbox_inches="tight")

