

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.ticker import FuncFormatter, FixedLocator

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Data for plotting (Years vs. Psychological Metrics)
years = np.array([1990, 1995, 2000, 2005, 2010, 2015, 2020])
stress_levels = np.array([5, 5.5, 6, 5.8, 5.6, 5.7, 5.75])
happiness_index = np.array([7, 6.8, 6.5, 6.6, 6.7, 6.9, 7.1])

# Axes Limits and Labels
xlabel_value = "Year"
ylabel_value = "Psychological Metrics"

yticks_values = [4, 5, 6, 7, 8]
yticks_labels = [
    "4",
    "5",
    "6",
    "7",
    "8",
]
ylim_values = [4.5, 8]

axvline_x = 2000

# Labels
label_stress = "Stress Levels"
label_happiness = "Happiness Index"
title = "Psychological Metrics Over Time"
legend_stress = "Stress"
legend_happiness = "Happiness"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plotting the data
plt.figure(figsize=(9, 6))
plt.plot(
    years,
    stress_levels,
    "o-",
    label=label_stress,
    color="#2ca02c",  # Green color for Stress Levels
    linewidth=1.4,
    markersize=4,
)
plt.plot(
    years,
    happiness_index,
    "s-",
    label=label_happiness,
    color="#8c564b",  # Brown color for Happiness Index
    linewidth=1.4,
    markersize=4,
)

# Setting the x-axis and y-axis to log scale
plt.xscale("log")
plt.yscale("log")

# Set y-axis to only display specific ticks and extend y-axis to leave space at top
plt.yticks(
    yticks_values,
    yticks_labels,
)
plt.ylim(ylim_values)  # Extend y-axis to leave some space above the highest value

# Disable the automatic grid for x-axis
plt.grid(True, which="both", ls="--", axis="y")  # Only enable y-axis grid

# Manually add a grid line for x=2000
plt.axvline(x=axvline_x, color="grey", linestyle="--", linewidth=0.5)

# Adjusting x-axis ticks to show significant years
plt.gca().xaxis.set_major_locator(FixedLocator(years))

# Formatting the x-axis and y-axis tick labels
plt.gca().xaxis.set_major_formatter(FuncFormatter(lambda value, _: "{:.0f}".format(value)))
plt.gca().yaxis.set_major_formatter(FuncFormatter(lambda y, _: "{:.0e}".format(y)))

# Adding labels and title
plt.xlabel(xlabel_value, fontsize=12)
plt.ylabel(ylabel_value, fontsize=12)
plt.title(title, fontsize=14)

# Adding a legend at the center right
plt.legend(loc="upper center",ncol=2, bbox_to_anchor=(0.5, 1), fontsize=18)

# ===================
# Part 4: Saving Output
# ===================
# Adjusting layout to reduce white space
plt.tight_layout()
plt.savefig("line_95.pdf", bbox_inches="tight")

