
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.gridspec as gridspec

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Climate Data for the pie chart
climate_factors = [
    "CO2 Emissions",
    "Deforestation",
    "Pollution",
    "Global Warming",
]
climate_factors_counts = [400, 200, 300, 100]

# Years and climate factor counts for the scatter plot
years_climate = [1990, 2000, 2010, 2020, 2030]
co2_emissions_count = [300, 400, 500, 600, 700]
deforestation_count = [200, 220, 240, 260, 280]
pollution_count = [100, 150, 200, 250, 300]
global_warming_count = [50, 100, 150, 200, 300]

# Axes Limits and Labels
title_1 = "Climate Factors Impacting Environment"

title_2 = "Trends in Climate Factors Over Years"
xlabel_value = "Year"
ylabel_value = "Impact Count"
ylim_values = [0, 800]
xlim_values = [1985, 2035]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and set up GridSpec
fig = plt.figure(figsize=(12, 6))
gs = gridspec.GridSpec(1, 2, width_ratios=[1, 1])

# Colors
colors_pie = ["#FF9999","#66B2FF","#99FF99","#FFD700"]

# Create pie chart on the left
ax1 = fig.add_subplot(gs[0])
wedges, texts, autotexts = ax1.pie(
    climate_factors_counts,
    labels=climate_factors,
    autopct="%1.1f%%",
    startangle=140,
    colors=colors_pie,
)
ax1.set_title(title_1)

# Create scatter plot on the right
ax2 = fig.add_subplot(gs[1])
scatter_co2_emissions = ax2.scatter(
    years_climate,
    co2_emissions_count,
    color="#FF9999",
    s=np.array(co2_emissions_count)/2,
    alpha=0.7,
    edgecolor="black",
    label="CO2 Emissions"
)
scatter_deforestation = ax2.scatter(
    years_climate,
    deforestation_count,
    color="#66B2FF",
    s=np.array(deforestation_count)/2,
    alpha=0.7,
    edgecolor="black",
    label="Deforestation"
)
scatter_pollution = ax2.scatter(
    years_climate,
    pollution_count,
    color="#99FF99",
    s=np.array(pollution_count)/2,
    alpha=0.7,
    edgecolor="black",
    label="Pollution"
)
scatter_global_warming = ax2.scatter(
    years_climate,
    global_warming_count,
    color="#FFD700",
    s=np.array(global_warming_count)/2,
    alpha=0.7,
    edgecolor="black",
    label="Global Warming"
)

ax2.set_title(title_2)
ax2.set_xlabel(xlabel_value)
ax2.set_ylabel(ylabel_value)
ax2.grid(True)
ax2.set_ylim(ylim_values)
ax2.set_xlim(xlim_values)

# Add text labels to scatter points
for i, year in enumerate(years_climate):
    ax2.annotate(f'{year}', (year, co2_emissions_count[i]), textcoords="offset points", xytext=(0,10), ha='center')

# Add legend to the scatter plot
ax2.legend()

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save plot
plt.tight_layout()
plt.savefig("multidiff_39.pdf", bbox_inches="tight")
