
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.gridspec as gridspec
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(42)

# Sample data to represent countries
countries = ["USA", "Germany", "China", "India", "France"]
gdp_per_capita = [65000, 50000, 10000, 2000, 40000]  # Average GDP per capita for each country
happiness_index = [7.0, 6.5, 5.5, 4.0, 6.8]  # Happiness index out of 10

# Data for violin plot; income distribution (in thousands)
income_distribution = np.random.normal(loc=[65, 50, 10, 2, 40], scale=[10, 8, 5, 1, 7], size=(300, 5))

# Labels and titles
ax0xlabel = "GDP per Capita ($)"
ax0ylabel = "Happiness Index"
ax0title = "Happiness Index vs GDP per Capita by Country"
ax1xticks = range(len(countries))
ax1xlabel = "Countries"
ax1ylabel = "Income Distribution (in $1000s)"
ax1title = "Income Distribution Across Countries"

x = np.linspace(-10, 100, 300)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create 1x2 subplot layout
fig = plt.figure(figsize=(12, 7))
gs = gridspec.GridSpec(1, 2, width_ratios=[1, 2])

# Scatter plot on the left
ax0 = fig.add_subplot(gs[0])
sc = ax0.scatter(
    gdp_per_capita, happiness_index, s=100, marker='o', c=np.linspace(0.1, 0.9, len(countries)), cmap="winter"
)
for i, country in enumerate(countries):
    ax0.text(gdp_per_capita[i], happiness_index[i] + 0.05, country, fontsize=9, style='italic')
ax0.set_xlabel(ax0xlabel)
ax0.set_ylabel(ax0ylabel)
ax0.set_title(ax0title)

# Violin plot on the right
ax1 = fig.add_subplot(gs[1])

# Creating half-violins
for i, country in enumerate(countries):
    kde = gaussian_kde(income_distribution[:, i])
    y = kde(x)
    max_y = max(y)
    ax1.fill_betweenx(x, -y / max_y * 0.5 + i, i, color="lightgreen", alpha=0.5)
    ax1.fill_betweenx(x, y / max_y * 0.5 + i, i, color="green", alpha=0.5)

ax1.set_xticks(ax1xticks)
ax1.set_xticklabels(countries)
ax1.set_xlabel(ax1xlabel)
ax1.set_ylabel(ax1ylabel)
ax1.set_title(ax1title)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
# Show plot
plt.savefig("multidiff_41.pdf", bbox_inches="tight")
