
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.gridspec as gridspec
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Sample data to represent renewable energy sources
energy_sources = ["Solar", "Wind", "Hydro", "Geothermal", "Biomass"]
costs = [40, 30, 20, 50, 45]  # Average cost ($/MWh) for each energy source
efficiency = [18, 35, 45, 50, 30]  # Efficiency percentage for each source

# Data for violin plot; daily energy output (MWh), different distribution means
energy_output_data = np.random.normal(loc=[300, 500, 700, 600, 400], scale=50, size=(50, 5))

ax0xlabel = "Average Cost ($/MWh)"
ax0ylabel = "Efficiency (%)"
ax0title = "Energy Source Efficiency vs Cost"
ax1xticks = range(len(energy_sources))
ax1xlabel = "Energy Sources"
ax1ylabel = "Daily Energy Output (MWh)"
ax1title = "Daily Energy Output Distribution Across Energy Sources"
x = np.linspace(100, 900, 300)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create 1x2 subplot layout
fig = plt.figure(figsize=(12, 6))
gs = gridspec.GridSpec(1, 2, width_ratios=[1, 2])

# Scatter plot on the left
ax0 = fig.add_subplot(gs[0])
sc = ax0.scatter(
    costs, efficiency, s=100, c=np.linspace(0.1, 0.9, len(energy_sources)), cmap="plasma"
)
for i, source in enumerate(energy_sources):
    ax0.text(costs[i]-0.5, efficiency[i] - 1, source, fontsize=9)
ax0.set_xlabel(ax0xlabel)
ax0.set_ylabel(ax0ylabel)
ax0.set_title(ax0title)

# Violin plot on the right
ax1 = fig.add_subplot(gs[1])

# Creating half-violins with different colors for each energy source
colors = ["#1f77b4", "#ff7f0e", "#2ca02c", "#d62728", "#9467bd"]
for i, source in enumerate(energy_sources):
    kde = gaussian_kde(energy_output_data[:, i])
    y = kde(x)
    max_y = max(y)
    ax1.fill_betweenx(x, -y / max_y * 0.5 + i, i, color=colors[i], alpha=0.5)
    ax1.fill_betweenx(x, y / max_y * 0.5 + i, i, color=colors[i], alpha=0.7)

ax1.set_xticks(ax1xticks)
ax1.set_xticklabels(energy_sources)
ax1.set_xlabel(ax1xlabel)
ax1.set_ylabel(ax1ylabel)
ax1.set_title(ax1title)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
# Show plot
plt.savefig("multidiff_43.pdf", bbox_inches="tight")
