
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(50)
# Data for the first plot (bar plot with white circles)
projects = [
    "ProjA", "ProjB", "ProjC", "ProjD", 
    "ProjE", "ProjF", "ProjG", "ProjH",
    "ProjI", "ProjJ", "ProjK", "ProjL",
    "ProjM", "ProjN", "ProjO", "ProjP"
]
n_projects = len(projects)
lines_of_code = np.random.uniform(2000, 15000, n_projects)
defect_density = np.random.uniform(0.5, 5.0, n_projects)
sorted_indices = np.argsort((lines_of_code + defect_density) / 2)
projects = np.array(projects)[sorted_indices]
lines_of_code = lines_of_code[sorted_indices]
defect_density = defect_density[sorted_indices]

categories = ["Frontend", "Backend", "Database", "DevOps"]

# Data for the second plot (error bar plot with legend)
team_performance = [np.random.randint(70, 90, size=5) for _ in range(len(categories))]
variability = np.random.randint(2, 7, size=(len(categories), 5))

# Data for the third plot (scatter plot with trend line and correct text)
costs = [
    np.linspace(10000, 50000, 5) + 5000 * i + np.random.randint(-5000, 5000, 5)
    for i in range(len(categories))
]
efficiencies = [
    np.linspace(80, 95, 5) - 2 * i + np.random.randint(-5, 5, 5)
    for i in range(len(categories))
]
scatterlabels = ["Estimated", "Measured"]
xlabels = ["Lines of Code", "Team Performance (%)", "Development Cost ($)"]
ylabels = ["Projects", "Performance (%)", "Efficiency (%)"]

ax2textlabel = "Category: DevOps"
title = "Software Development Projects Overview"
text_j = 40000
text_i = 90
legendtitle="Metric Type"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Adjusting figure size to match the original image's dimensions
fig, axs = plt.subplots(1, 3, figsize=(12, 5))

# Colors and assignments for the first plot
colors = ["#f39c12", "#3498db", "#2ecc71", "#e74c3c"]

# First plot (bar plot with white circles)
axs[0].grid(axis="x", zorder=0)
# Plotting the lines first
for i in range(n_projects):
    color = colors[i // 4]
    axs[0].plot(
        [defect_density[i], lines_of_code[i] / 1000], [i, i], color=color, lw=6, zorder=1
    )
# Plotting the points
axs[0].scatter(
    lines_of_code / 1000,
    range(n_projects),
    color="black",
    label=scatterlabels[0],
    zorder=2,
    s=80,
)  # Black dots for 'Estimated'
axs[0].scatter(
    defect_density,
    range(n_projects),
    color="white",
    edgecolors="black",
    label=scatterlabels[1],
    zorder=2,
    s=80,
)  # White dots for 'Measured'
axs[0].legend(loc="lower right", title=legendtitle)
axs[0].set_xlabel(xlabels[0])
axs[0].set_ylabel(ylabels[0])
axs[0].set_yticks(range(n_projects))
axs[0].set_yticklabels(projects[::-1])

# Second plot (error bar plot with legend)
for i in range(len(categories)):
    team_perf = team_performance[i]
    var = variability[i]
    axs[1].errorbar(
        np.linspace(1, 5, 5) * 10,
        team_perf,
        yerr=var,
        fmt="o",
        capsize=3,
        color=colors[i],
    )
axs[1].set_xlabel(xlabels[1])
axs[1].set_ylabel(ylabels[1])
axs[1].grid()

# Third plot (scatter plot with trend line and correct text)
for i in range(len(categories)):
    cost = costs[i]
    efficiency = efficiencies[i]
    error = np.random.randint(5, 10, size=5)
    axs[2].errorbar(
        cost,
        efficiency,
        yerr=error,
        xerr=error + 1000,
        fmt="o",
        capsize=3,
        color=colors[i],
    )
# Plotting the trend line
axs[2].plot(
    np.linspace(10000, 50000, 5), np.linspace(75, 95, 5), color="black", lw=2, ls="-"
)
axs[2].set_xlabel(xlabels[2])
axs[2].set_ylabel(ylabels[2])
axs[2].text(
    text_j,
    text_i,
    ax2textlabel,
    fontsize=8,
    color="black",
    ha="right",
    va="bottom",
    bbox=dict(facecolor="white", edgecolor="black"),
)
axs[2].grid()

# add legend to the whole figure and set colormaps
legend_handles = [
    mlines.Line2D(
        [], [], color=color, marker="s", linestyle="None", markersize=10, label=label
    )
    for label, color in zip(categories, colors)
]
fig.legend(
    handles=legend_handles,
    loc="upper center",
    title=title,
    ncol=len(categories),
    bbox_to_anchor=(0.5, 1.15),
    facecolor="white",
)

# ===================
# Part 4: Saving Output
# ===================
# Show plot
plt.tight_layout()
plt.savefig("multidiff_50.pdf", bbox_inches="tight")
