

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Sample data for the line plot (monthly average temperatures)
months = np.array([1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12])
temperatures = 10 + 10 * np.sin(2 * np.pi * (months - 3) / 12) + np.random.normal(0, 1, 12)
temperature_error = np.random.normal(0.5, 0.2, 12)

# Sample data for the box plot (seasonal rainfall in mm)
rainfall_data = [np.random.normal(50 + 20 * season, 10, 100) for season in range(4)]
seasons = [1, 2, 3, 4]
# Add scatter data points for seasonal rainfall data
rainfall_scatter_data = np.random.normal(60, 5, len(seasons))

# Labels and Titles
titles = ["(a) Monthly Average Temperatures", "(b) Seasonal Rainfall"]
xlabels = ["Month", "Season"]
ylabels = ["Temperature (°C)", "Rainfall (mm)"]
xtickslabels = [
    ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"],
    ["Winter", "Spring", "Summer", "Autumn"],
]
yticks = [np.arange(-10, 31, 5), np.arange(0, 151, 25)]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and axes
fig, axs = plt.subplots(2, 1, figsize=(6, 8))

# Line plot
axs[0].errorbar(
    months,
    temperatures,
    yerr=temperature_error,
    fmt="o-",
    ecolor="lightblue",
    markersize=8,
    linewidth=2,
    color="blue",
    linestyle='--',
    marker='s'
)
axs[0].fill_between(months, temperatures - temperature_error, temperatures + temperature_error, color="lightblue", alpha=0.5)
axs[0].set_title(titles[0])
axs[0].set_xlabel(xlabels[0])
axs[0].set_ylabel(ylabels[0])
axs[0].grid(True)
axs[0].set_xticks(months)
axs[0].set_xticklabels(xtickslabels[0])
axs[0].set_yticks(yticks[0])

# Box plot
axs[1].boxplot(
    rainfall_data,
    positions=seasons,
    widths=0.5,
    showfliers=False,
    boxprops=dict(color="darkgreen", linewidth=2),
    medianprops=dict(color="green", linewidth=2),
    whiskerprops=dict(color="darkgreen", linewidth=2),
    capprops=dict(color="darkgreen", linewidth=2),
)
axs[1].set_title(titles[1])
axs[1].set_xlabel(xlabels[1])
axs[1].set_ylabel(ylabels[1])
axs[1].yaxis.grid(True)
axs[1].set_xticks(seasons)
axs[1].set_xticklabels(xtickslabels[1])
axs[1].set_yticks(yticks[1])

# Scatter plot overlay
axs[1].scatter(seasons, rainfall_scatter_data, marker="^", color="darkgreen", s=100, zorder=3)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout
plt.tight_layout()

plt.savefig("multidiff_65.pdf", bbox_inches="tight")

