
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# ErrorBar Plot Data
# Types of Therapies
therapies = ["CBT", "Psychoanalysis", "Behavioral", "Humanistic"]
# Hypothetical effectiveness rates (percentage)
effectiveness_rates = [75, 60, 70, 80]
# Standard errors for the above effectiveness rates
effectiveness_errors = [5, 7, 6, 8]

# ErrorPoint Plot Data
# Psychological Constructs
psych_constructs = ["Anxiety", "Depression", "Stress", "Self-esteem"]
# Randomly generated prevalence rates and standard deviations
prevalence_rates = np.random.uniform(0.3, 0.8, len(psych_constructs))
std_devs_prevalence = np.random.uniform(0.02, 0.05, len(psych_constructs))
dataset_mean_prevalence = np.mean(prevalence_rates)

# Labels and Titles
titles = ["Effectiveness of Therapies", "Prevalence of Psychological Constructs"]
ylabels = ["Effectiveness (%)", "Prevalence Rate"]
ylim = [0, 1.2]
xlabel_constructs = "Psychological Constructs"
legendlabels = ["Prevalence Rates", "Mean Prevalence Rate"]
subtitle = "Psychology Data Visualizations"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Colors
bar_color = "#D2B48C"
error_color_bar = "brown"
point_color = "#8B4513"
error_color_point = "#8B4513"
hline_color = "grey"

# Create figure and axes for the subplots
fig, axes = plt.subplots(2, 1, figsize=(10, 12))
fig.suptitle(subtitle, fontsize=16)

# ErrorBar Plot
axes[0].bar(
    therapies, effectiveness_rates, yerr=effectiveness_errors, color=bar_color, capsize=5, ecolor=error_color_bar
)
axes[0].set_title(titles[0])
axes[0].set_ylabel(ylabels[0])
axes[0].grid(True, linestyle='--', linewidth=0.5)

# ErrorPoint Plot
axes[1].errorbar(
    psych_constructs,
    prevalence_rates,
    yerr=std_devs_prevalence,
    fmt="D",
    color=point_color,
    ecolor=error_color_point,
    capsize=5,
    ms=8,
    linestyle='None'
)
axes[1].axhline(y=dataset_mean_prevalence, color=hline_color, linestyle="--")
axes[1].set_title(titles[1])
axes[1].set_ylabel(ylabels[1])
axes[1].set_xlabel(xlabel_constructs)
axes[1].set_ylim(ylim)
axes[1].legend(legendlabels, loc='upper left')
axes[1].grid(True, linestyle='--', linewidth=0.5)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout to avoid overlap and save the figure
plt.tight_layout(rect=[0, 0.03, 1, 0.97])
plt.savefig("multidiff_80.pdf", bbox_inches="tight")
