

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
# Seed for reproducibility
import numpy as np

np.random.seed(0)

# Generating synthetic data for the violin plot
# Average number of philosophical books read per year by two student groups
books_undergrad = np.random.normal(12, 3, 150)  # Books read by undergraduates
books_grad = np.random.normal(18, 4, 150)  # Books read by graduate students

# Generating synthetic data for the density plot
# Response times in seconds to different philosophical questions
ethics_response_time = np.random.normal(30, 10, 1000)  # Ethics questions response time
metaphysics_response_time = np.random.normal(40, 12, 1000)  # Metaphysics questions response time
epistemology_response_time = np.random.normal(35, 11, 1000)  # Epistemology questions response time

xs = np.linspace(0, 100, 200)

labels = ["Ethics", "Metaphysics", "Epistemology"]
titles = ["Books Read Per Year", "Philosophical Question Response Time Distribution"]
ylabels = ["Number of Books", "Density"]
ax1xticks = [1, 2]
ax1xtickslabels = ["Undergraduates", "Graduates"]
ax2xlabel = "Response Time (seconds)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Creating the figure and axes
fig, axes = plt.subplots(nrows=1, ncols=2, figsize=(10, 5))

# Violin plot
violin_data = [books_undergrad, books_grad]
axes[0].violinplot(violin_data, showmeans=False, showmedians=True)
axes[0].set_title(titles[0])
axes[0].set_ylabel(ylabels[0])
axes[0].set_xticks(ax1xticks)
axes[0].set_xticklabels(ax1xtickslabels)
axes[0].grid(True)

# Density plot
colors = ["#1f77b4", "#ff7f0e", "#2ca02c"]  # Custom colors

for data, color, label in zip([ethics_response_time, metaphysics_response_time, epistemology_response_time], colors, labels):
    density = gaussian_kde(data)
    axes[1].fill_between(xs, density(xs), color=color, alpha=0.5, label=label)
axes[1].set_title(titles[1])
axes[1].set_xlabel(ax2xlabel)
axes[1].set_ylabel(ylabels[1])
axes[1].legend()

# ===================
# Part 4: Saving Output
# ===================
# Adjusting layout for better visual appearance
plt.tight_layout()
plt.savefig("multidiff_83.pdf", bbox_inches="tight")
