

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Seed for reproducibility
np.random.seed(0)

# Generating synthetic data for the violin plot (Sales Amounts)
electronics_sales = np.random.normal(1000, 200, 150)  # Electronics department sales
clothing_sales = np.random.normal(750, 180, 150)  # Clothing department sales

# Generating synthetic data for the density plot (Customer Satisfaction Scores)
north_scores = np.random.normal(70, 15, 1000)  # North region customer satisfaction scores
south_scores = np.random.normal(65, 10, 1000)  # South region customer satisfaction scores
west_scores = np.random.normal(75, 15, 1000)  # West region customer satisfaction scores

# Defining x-axis range for density plots
xs = np.linspace(20, 120, 200)

# Labels and titles for the plots
labels = ["North", "South", "West"]
titles = ["Monthly Sales Distribution by Department", "Customer Satisfaction Scores by Region"]
ylabels = ["Sales Amount ($)", "Density"]
ax1xticks = [1, 2]
ax1xtickslabels = ["Electronics", "Clothing"]
ax2xlabel = "Customer Satisfaction Score"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Creating the figure and axes
fig, axes = plt.subplots(nrows=1, ncols=2, figsize=(12, 6))

# Violin plot
violin_data = [electronics_sales, clothing_sales]
axes[0].violinplot(violin_data, showmeans=False, showmedians=True, 
                   showextrema=False, bw_method=0.5)
axes[0].set_title(titles[0], fontsize=14, weight='bold')
axes[0].set_ylabel(ylabels[0], fontsize=12)
axes[0].set_xticks(ax1xticks)
axes[0].set_xticklabels(ax1xtickslabels, fontsize=10)
axes[0].grid(True)

# Density plot
colors = ["#1f77b4", "#ff7f0e", "#2ca02c"]  # Custom colors for each region
line_styles = ['-', '--', '-.']
for data, color, label, linestyle in zip([north_scores, south_scores, west_scores], colors, labels, line_styles):
    density = gaussian_kde(data)
    axes[1].plot(xs, density(xs), color=color, linestyle=linestyle, linewidth=2, label=label)
    axes[1].fill_between(xs, density(xs), color=color, alpha=0.2)
axes[1].set_title(titles[1], fontsize=14, weight='bold')
axes[1].set_xlabel(ax2xlabel, fontsize=12)
axes[1].set_ylabel(ylabels[1], fontsize=12)
axes[1].legend(fontsize=10)
axes[1].grid(True)

# ===================
# Part 4: Saving Output
# ===================
# Adjusting layout for better visual appearance
plt.tight_layout()
plt.savefig("multidiff_85.pdf", bbox_inches="tight")

