
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.gridspec import GridSpec

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(90)

# ========== Bar with Error Data (Mean Stress Levels) ==========
# Professions
professions = ["Teacher", "Nurse", "Engineer", "Artist"]
# Mean stress levels (in arbitrary units)
stress_levels = [7.5, 8.0, 6.0, 5.5]
# Error margins
stress_errors = [0.5, 0.8, 0.6, 0.4]

# ========== Scatter with Error Data (Cognitive Performance Scores) ==========
# Age Groups
age_groups = ["20-30", "31-40", "41-50", "51-60"]
# Cognitive performance scores (arbitrary units)
cognitive_scores = np.random.uniform(70, 90, len(age_groups))
# Standard deviations
cognitive_std_devs = np.random.uniform(2, 5, len(age_groups))

# ========== Line with Error Data (Mental Health Symptoms) ==========
# Symptoms
mental_symptoms = ["Anxiety", "Depression", "Stress", "Fatigue"]
# Symptom levels over time (in arbitrary units)
symptom_2020 = np.random.uniform(30, 50, len(mental_symptoms))
symptom_2021 = np.random.uniform(25, 45, len(mental_symptoms))
# Error
symptom_error = np.random.uniform(2, 4, len(mental_symptoms))

# Titles and Labels
titles = [
    "Mean Stress Levels by Profession",
    "Cognitive Performance Scores by Age Group",
    "Mental Health Symptoms Over Time",
]
ylabels = ["Stress Level (units)", "Cognitive Score (units)", "Symptom Level (units)"]
ax2ylim = [65, 95]
ax3labels = ["2020", "2021"]

legend_title = "Year"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and grid layout
fig = plt.figure(figsize=(10, 10))
gs = GridSpec(2, 2, figure=fig)

# Bar with Error Plot
ax1 = fig.add_subplot(gs[0, :])
ax1.bar(professions, stress_levels, yerr=stress_errors, color="steelblue", capsize=5, ecolor="grey")
ax1.set_title(titles[0])
ax1.set_ylabel(ylabels[0])
ax1.grid(True)

# Scatter with Error Plot
ax2 = fig.add_subplot(gs[1, 0])
ax2.errorbar(
    age_groups,
    cognitive_scores,
    yerr=cognitive_std_devs,
    fmt="o",
    color="orange",
    ecolor="grey",
    capsize=5,
    ms=10,
)
ax2.set_title(titles[1])
ax2.set_ylabel(ylabels[1])
ax2.set_ylim(ax2ylim)
ax2.grid(True)

# Line with Error Plot
ax3 = fig.add_subplot(gs[1, 1])
ax3.errorbar(
    mental_symptoms,
    symptom_2020,
    yerr=symptom_error,
    fmt="^-",
    color="green",
    ecolor="grey",
    capsize=5,
    label=ax3labels[0],
)
ax3.errorbar(
    mental_symptoms,
    symptom_2021,
    yerr=symptom_error,
    fmt="s-",
    color="blue",
    ecolor="grey",
    capsize=5,
    label=ax3labels[1],
)
ax3.set_title(titles[2])
ax3.set_ylabel(ylabels[2])
ax3.legend(title=legend_title)
ax3.grid(True)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the figure
plt.tight_layout()
plt.savefig("multidiff_90.pdf", bbox_inches="tight")
