

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde
from matplotlib.gridspec import GridSpec

# ===================
# Part 2: Data Preparation
# ===================
# Seed for reproducibility
import numpy as np

np.random.seed(0)

# Generate synthetic weather data
# Temperature data for Winter and Summer
winter_temps = np.random.normal(0, 5, 500)  # Winter temperatures
summer_temps = np.random.normal(25, 5, 500)  # Summer temperatures

# Daily rainfall data for two regions
region1_rainfall = np.random.gamma(2, 2, 1000)  # Region 1 rainfall
region2_rainfall = np.random.gamma(3, 1.5, 1000)  # Region 2 rainfall

# Average monthly humidity over time
time_period = np.linspace(0, 5, 1000)  # Time in years
avg_humidity = 60 + 10 * np.sin(2 * np.pi * time_period / 5)  # Sine wave pattern

# Labels and titles for the plots
ax1labels = ["Winter Temperatures", "Summer Temperatures"]
titles = [
    "Temperature Distribution by Season",
    "Rainfall Distribution by Region",
    "Average Monthly Humidity Over Time",
]
xlabels = ["Temperature (°C)", "Time (Years)"]
ylabels = ["Frequency", "Rainfall (mm)", "Humidity (%)"]
ax2xtickslabels = ["Region 1", "Region 2"]
ax2xticks = [1, 2]
bins = np.linspace(-10, 40, 21)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Define color variables
hist_colors = ["#3b75af", "#88c0d0"]
violin_color = "green"

# Create the figure using GridSpec
fig = plt.figure(figsize=(10, 10))
gs = GridSpec(2, 2, figure=fig)

# Histogram plot across top (1,1 and 1,2)
ax1 = fig.add_subplot(gs[0, :])
# Define bin edges and width to align bars side by side

ax1.hist(
    winter_temps,
    bins=bins,
    alpha=0.7,
    label=ax1labels[0],
    color=hist_colors[0],
    edgecolor="black",
)
ax1.hist(
    summer_temps,
    bins=bins,
    alpha=0.7,
    label=ax1labels[1],
    color=hist_colors[1],
    edgecolor="black",
)
ax1.set_title(titles[0])
ax1.set_xlabel(xlabels[0])
ax1.set_ylabel(ylabels[0])
ax1.legend()

# Violin plot on bottom left (2,1)
ax2 = fig.add_subplot(gs[1, 0])
ax2.violinplot([region1_rainfall, region2_rainfall], showmeans=False, showmedians=True)
ax2.set_title(titles[1])
ax2.set_ylabel(ylabels[1])
ax2.set_xticks(ax2xticks)
ax2.set_xticklabels(ax2xtickslabels)
ax2.grid(True)

# Fill between plot on bottom right (2,2)
ax3 = fig.add_subplot(gs[1, 1])
ax3.fill_between(time_period, avg_humidity, color=violin_color, alpha=0.2)
ax3.set_title(titles[2])
ax3.set_xlabel(xlabels[1])
ax3.set_ylabel(ylabels[2])

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for a better visual appearance
plt.tight_layout()
plt.savefig("multidiff_93.pdf", bbox_inches="tight")

