

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde
from matplotlib.gridspec import GridSpec

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Seed for reproducibility
np.random.seed(0)
# Generate synthetic environmental science data
# Temperature counts at different times of day
morning_temps = np.random.normal(15, 5, 500)  # Morning temperature in °C
evening_temps = np.random.normal(20, 5, 500)  # Evening temperature in °C

# Humidity levels through different times of day
morning_humidity = np.random.normal(80, 10, 1000)  # Morning humidity in %
evening_humidity = np.random.normal(60, 15, 1000)  # Evening humidity in %

# Elevation data in a hiking route
hiking_route_distance = np.linspace(0, 10, 1000)  # Distance in km
hiking_elevation_changes = np.sin(np.linspace(0, 3, 1000)) * 100 + 100

ax1labels = ["Morning Temperatures", "Evening Temperatures"]
titles = [
    "Temperature Distribution Throughout the Day",
    "Humidity Levels During the Day",
    "Elevation Changes Along the Hiking Route",
]
xlabels = ["Temperature (°C)", "Distance (km)"]
ylabels = ["Frequency", "Humidity (%)", "Elevation (m)"]
ax2xtickslabels = ["Morning", "Evening"]
ax2xticks = [1, 2]
bins = np.linspace(0, 35, 31)


# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the figure using GridSpec
fig = plt.figure(figsize=(10, 10))
gs = GridSpec(2, 2, figure=fig)

# Histogram plot across top (1,1 and 1,2)
ax1 = fig.add_subplot(gs[0, :])

ax1.hist(
    morning_temps,
    bins=bins,
    alpha=0.7,
    label=ax1labels[0],
    color="#FFB3BA",
    edgecolor="black",
)
ax1.hist(
    evening_temps,
    bins=bins,
    alpha=0.7,
    label=ax1labels[1],
    color="#FFDFBA",
    edgecolor="black",
)
ax1.set_title(titles[0])
ax1.set_xlabel(xlabels[0])
ax1.set_ylabel(ylabels[0])
ax1.legend()

# Violin plot on bottom left (2,1)
ax2 = fig.add_subplot(gs[1, 0])
ax2.violinplot([morning_humidity, evening_humidity], showmeans=False, showmedians=True)
ax2.set_title(titles[1])
ax2.set_ylabel(ylabels[1])
ax2.set_xticks(ax2xticks)
ax2.set_xticklabels(ax2xtickslabels)
ax2.grid(True)

# Fill between plot on bottom right (2,2)
ax3 = fig.add_subplot(gs[1, 1])
ax3.fill_between(hiking_route_distance, hiking_elevation_changes, color="#BAE1FF", alpha=0.2)
ax3.set_title(titles[2])
ax3.set_xlabel(xlabels[1])
ax3.set_ylabel(ylabels[2])

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for a better visual appearance
plt.tight_layout()
plt.savefig("multidiff_94.pdf", bbox_inches="tight")

