

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.gridspec as gridspec

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Data for Picture in Picture bar plot
internet_usage_known_sites = np.array([5000, 7000, 8000, 12000, 15000, 13000, 9000, 6000, 4000, 3000, 2000])
internet_usage_unknown_sites = np.array([2000, 3000, 4000, 6000, 5000, 4000, 3000, 2000, 1500, 1000, 500])
age_groups = np.linspace(10, 60, 11)

# Data for the heatmap
policy_approval_ratings = np.array([
    [55, 65, 70, 60, 75, 80],
    [50, 60, 65, 55, 70, 75],
    [45, 55, 60, 50, 65, 70],
    [48, 58, 63, 53, 68, 73],
    [52, 62, 67, 57, 72, 77],
])
x_labels = ["Policy A", "Policy B", "Policy C", "Policy D", "Policy E", "Policy F"]
y_labels = ["Region 1", "Region 2", "Region 3", "Region 4", "Region 5"]
bar_labels = ["Known Sites", "Unknown Sites"]
xlabels = ["Age Groups", "Public Policies"]
ylabels = ["Number of Users", "Regions"]
cbarlabel = "Approval Ratings"
insetaxes = [0.30, 0.6, 0.1, 0.2]
insetxlim = [30, 60]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with GridSpec
fig = plt.figure(figsize=(12, 5))
gs = gridspec.GridSpec(1, 2, width_ratios=[1, 1.5])

# Picture in Picture bar plot
ax0 = plt.subplot(gs[0])
ax0.bar(
    age_groups,
    internet_usage_known_sites,
    width=3,
    color="#FF5733",  # Orange for known sites
    align="center",
    label=bar_labels[0],
    edgecolor="black",
)
ax0.bar(
    age_groups,
    internet_usage_unknown_sites,
    width=3,
    color="#33C3FF",  # Light blue for unknown sites
    align="center",
    bottom=internet_usage_known_sites,
    label=bar_labels[1],
    edgecolor="black",
)
ax0.set_xlabel(xlabels[0])
ax0.set_ylabel(ylabels[0])
ax0.legend(loc="upper right")
ax0.grid(True)

# Add inset
ax_inset = fig.add_axes(insetaxes)
ax_inset.bar(
    age_groups[5:],
    internet_usage_known_sites[5:],
    width=3,
    color="#FF5733",
    align="center",
    edgecolor="black",
)
ax_inset.bar(
    age_groups[5:],
    internet_usage_unknown_sites[5:],
    width=3,
    color="#33C3FF",
    align="center",
    bottom=internet_usage_known_sites[5:],
    edgecolor="black",
)
ax_inset.set_xlim(insetxlim)

# Heatmap plot
ax1 = plt.subplot(gs[1])
cmap = plt.cm.YlGnBu
norm = plt.Normalize(vmin=policy_approval_ratings.min(), vmax=policy_approval_ratings.max())
heatmap = ax1.imshow(policy_approval_ratings, cmap=cmap, norm=norm, aspect="auto")

# Add color bar
cbar = plt.colorbar(heatmap, ax=ax1, orientation="vertical", pad=0.1)
cbar.set_label(cbarlabel)

# Set x and y labels
ax1.set_xticks(np.arange(len(x_labels)))
ax1.set_yticks(np.arange(len(y_labels)))
ax1.set_xticklabels(x_labels, rotation=45)
ax1.set_yticklabels(y_labels)
ax1.set_xlabel(xlabels[1])
ax1.set_ylabel(ylabels[1])

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("multidiff_98.pdf", bbox_inches="tight")

