
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
# Data for the plot
portfolios_1 = ["Tech Stocks", "Healthcare Stocks", "Energy Stocks"]
start_bias_1 = [8.0, 7.5, 6.8]
stock_bias_1 = [-1.8, -2.0, -1.5]
start_bond_1 = [5.5, 6.0, 5.8]
bond_bias_1 = [1.5, 1.7, 1.4]
ax1_labels = ["Stock Bias\n(Diversification)", "Bond Bias\n(Risk Aversion)"]

portfolios_2 = ["Government Bonds", "Corporate Bonds", "Real Estate"]
start_bias_2 = [6.5, 7.0, 7.8]
stock_bias_2 = [-1.2, -1.7, -1.5]
start_bond_2 = [5.0, 6.2, 6.5]
bond_bias_2 = [1.3, 1.1, 1.2]
ax2_labels = ["Stock Bias\n(Risk Aversion)", "Bond Bias\n(Diversification)"]

# Plot titles and labels
xlabel = "Bias Scale"
ylabel = "Investment Portfolios"
supertitle = "Biases in Different Investment Portfolios"
title_1 = "Stock-Focused Portfolios"
title_2 = "Bond-Focused Portfolios"
legend_labels = ["Bias Type"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(12, 5))
fig.suptitle(supertitle, fontsize=16)

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# Colors for the arrows
out_group_color = "#FF6347"  # Tomato (Stock Bias)
in_group_color = "#4682B4"  # SteelBlue (Bond Bias)

# First subplot (portfolios_1)
for i, portfolio in enumerate(portfolios_1):
    # Stock bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(start_bias_1[i], i + offset * 3 / 2),
        xytext=(start_bias_1[i] + stock_bias_1[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color=out_group_color),
    )
    ax1.scatter(
        [start_bias_1[i], start_bias_1[i] + stock_bias_1[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{stock_bias_1[i]:.2f}",
        (start_bias_1[i] + stock_bias_1[i], i + offset * 1.75),
        color=out_group_color,
        ha="right",
        va="center",
    )

    # Bond bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(start_bond_1[i], i + offset / 2),
        xytext=(start_bond_1[i] + bond_bias_1[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<-", color=in_group_color),
    )
    ax1.scatter(
        [start_bond_1[i], start_bond_1[i] + bond_bias_1[i]],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{bond_bias_1[i]:.2f}",
        (start_bond_1[i] + bond_bias_1[i], i + offset * 0.75),
        color=in_group_color,
        ha="left",
        va="center",
    )

# Second subplot (portfolios_2)
for i, portfolio in enumerate(portfolios_2):
    ax2.annotate(
        "",
        xy=(start_bias_2[i], i + offset * 3 / 2),
        xytext=(start_bias_2[i] + stock_bias_2[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color=out_group_color),
    )
    ax2.scatter(
        [start_bias_2[i], start_bias_2[i] + stock_bias_2[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{stock_bias_2[i]:.2f}",
        (start_bias_2[i] + stock_bias_2[i], i + offset * 1.75),
        color=out_group_color,
        ha="right",
        va="center",
    )

    ax2.annotate(
        "",
        xy=(start_bond_2[i], i + offset / 2),
        xytext=(start_bond_2[i] + bond_bias_2[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<-", color=in_group_color),
    )
    ax2.scatter(
        [start_bond_2[i], start_bond_2[i] + bond_bias_2[i]],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{bond_bias_2[i]:.2f}",
        (start_bond_2[i] + bond_bias_2[i], i + offset * 0.75),
        color=in_group_color,
        ha="left",
        va="center",
    )

# Set y-axis limits
ax1.set_ylim(0, len(portfolios_1))
ax2.set_ylim(0, len(portfolios_2))

# Set x-axis limits uniformly
ax1.set_xlim(4, 10)
ax2.set_xlim(4, 10)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(portfolios_1))])
ax1.set_yticklabels(portfolios_1,rotation=45)
ax2.set_yticks([i + offset for i in range(len(portfolios_2))])
ax2.set_yticklabels(portfolios_2)
ax2.yaxis.tick_right()
ax2.yaxis.set_label_position("right")

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(portfolios_1))], minor=True)
ax2.set_yticks([i for i in range(len(portfolios_2))], minor=True)
ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="black")
ax2.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="black")

# Add x-axis grid lines and set gap is 1
ax1.xaxis.set_major_locator(plt.MultipleLocator(1))
ax2.xaxis.set_major_locator(plt.MultipleLocator(1))
ax1.grid(axis="x", linestyle="--", linewidth=0.5)
ax2.grid(axis="x", linestyle="--", linewidth=0.5)

# Create arrow-shaped legend entries with a line that aligns with the arrowhead
red_arrow_1 = mlines.Line2D(
    [],
    [],
    color=out_group_color,
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax1_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
blue_arrow_1 = mlines.Line2D(
    [],
    [],
    color=in_group_color,
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax1_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(
    handles=[red_arrow_1, blue_arrow_1],
    bbox_to_anchor=(0.45, 0),
    ncol=2,
    title=legend_labels[0],
)

red_arrow_2 = mlines.Line2D(
    [],
    [],
    color=out_group_color,
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax2_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
blue_arrow_2 = mlines.Line2D(
    [],
    [],
    color=in_group_color,
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax2_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[red_arrow_2, blue_arrow_2], bbox_to_anchor=(0.85, 0), ncol=2)

# Set individual titles for the subplots
ax1.set_title(title_1)
ax2.set_title(title_2)

# Add x and y axis labels
fig.text(0.5, 0.04, xlabel, ha="center", va="center")
fig.text(-0.0, 0.5, ylabel, ha="center", va="center", rotation="vertical")

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout(rect=[0, 0.03, 1, 0.95])
plt.savefig("quiver_23.pdf", bbox_inches="tight")
