
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Data for the plot
# Domain: Media Efficiency and Audience Satisfaction Across Different Media Types
media_types = ["Social Media", "Television", "Print Media"]
efficiency_scores_media = [8.1, 7.6, 6.8]
efficiency_bias_media = [+2.2, +1.8, +1.4]
satisfaction_scores_media = [7.5, 7.1, 6.4]
satisfaction_bias_media = [-2.0, -1.7, -1.3]
media_labels = ["Efficiency Bias", "Satisfaction Bias"]

departments_media = ["Marketing", "Content Creation", "Distribution"]
efficiency_scores_departments_media = [8.0, 7.5, 6.9]
efficiency_bias_departments_media = [+2.4, +2.0, +1.6]
satisfaction_scores_departments_media = [7.6, 7.3, 6.7]
satisfaction_bias_departments_media = [-2.1, -1.8, -1.5]
departments_labels_media = ["Efficiency Bias", "Satisfaction Bias"]

# Text labels
xlabel = "Scores"
ylabel = "Media Types / Departments"
suptitle = "Media Efficiency and Audience Satisfaction"
title_media = "Media Types"
title_departments_media = "Departments"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Colors for the plot
efficiency_color = "royalblue"
satisfaction_color = "darkgrey"
dot_color = "black"

# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(12, 5))

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# First subplot (media types)
for i in range(len(media_types)):
    # Efficiency bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(efficiency_scores_media[i], i + offset * 3 / 2),
        xytext=(
            efficiency_scores_media[i] + efficiency_bias_media[i],
            i + offset * 3 / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="solid", color=efficiency_color),
    )
    ax1.scatter(
        [
            efficiency_scores_media[i],
            efficiency_scores_media[i] + efficiency_bias_media[i],
        ],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color=dot_color,
        s=10,
    )
    ax1.annotate(
        f"{efficiency_bias_media[i]:.2f}",
        (
            efficiency_scores_media[i] + efficiency_bias_media[i],
            i + offset * 1.75,
        ),
        color=efficiency_color,
        ha="left",
        va="center",
    )

    # Satisfaction bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(satisfaction_scores_media[i], i + offset / 2),
        xytext=(
            satisfaction_scores_media[i] + satisfaction_bias_media[i],
            i + offset / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="dashed", color=satisfaction_color),
    )
    ax1.scatter(
        [
            satisfaction_scores_media[i],
            satisfaction_scores_media[i] + satisfaction_bias_media[i],
        ],
        [i + offset / 2, i + offset / 2],
        color=dot_color,
        s=10,
    )
    ax1.annotate(
        f"{satisfaction_bias_media[i]:.2f}",
        (
            satisfaction_scores_media[i] + satisfaction_bias_media[i],
            i + offset * 0.75,
        ),
        color=satisfaction_color,
        ha="right",
        va="center",
    )

# Second subplot (departments)
for i in range(len(departments_media)):
    ax2.annotate(
        "",
        xy=(efficiency_scores_departments_media[i], i + offset * 3 / 2),
        xytext=(
            efficiency_scores_departments_media[i] + efficiency_bias_departments_media[i],
            i + offset * 3 / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="solid", color=efficiency_color),
    )
    ax2.scatter(
        [
            efficiency_scores_departments_media[i],
            efficiency_scores_departments_media[i] + efficiency_bias_departments_media[i],
        ],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color=dot_color,
        s=10,
    )
    ax2.annotate(
        f"{efficiency_bias_departments_media[i]:.2f}",
        (
            efficiency_scores_departments_media[i] + efficiency_bias_departments_media[i],
            i + offset * 1.75,
        ),
        color=efficiency_color,
        ha="left",
        va="center",
    )

    ax2.annotate(
        "",
        xy=(satisfaction_scores_departments_media[i], i + offset / 2),
        xytext=(
            satisfaction_scores_departments_media[i] + satisfaction_bias_departments_media[i],
            i + offset / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="dashed", color=satisfaction_color),
    )
    ax2.scatter(
        [
            satisfaction_scores_departments_media[i],
            satisfaction_scores_departments_media[i] + satisfaction_bias_departments_media[i],
        ],
        [i + offset / 2, i + offset / 2],
        color=dot_color,
        s=10,
    )
    ax2.annotate(
        f"{satisfaction_bias_departments_media[i]:.2f}",
        (
            satisfaction_scores_departments_media[i] + satisfaction_bias_departments_media[i],
            i + offset * 0.75,
        ),
        color=satisfaction_color,
        ha="right",
        va="center",
    )

# Set y-axis limits
ax1.set_ylim(0, len(media_types))
ax2.set_ylim(0, len(departments_media))

# Set x-axis limits uniformly
ax1.set_xlim(0, 12)
ax2.set_xlim(0, 12)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(media_types))])
ax1.set_yticklabels(media_types)
ax2.set_yticks([i + offset for i in range(len(departments_media))])
ax2.set_yticklabels(departments_media)
ax2.yaxis.tick_right()
ax2.yaxis.set_label_position("right")

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(media_types))], minor=True)
ax2.set_yticks([i for i in range(len(departments_media))], minor=True)
ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="gray")
ax2.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="gray")

# Add x-axis grid lines and set gap to 1
ax1.xaxis.set_major_locator(plt.MultipleLocator(1))
ax2.xaxis.set_major_locator(plt.MultipleLocator(1))
ax1.grid(axis="x", linestyle="--", linewidth=0.5)
ax2.grid(axis="x", linestyle="--", linewidth=0.5)

# Create arrow-shaped legend entries with a line that aligns with the arrowhead
blue_arrow = mlines.Line2D(
    [],
    [],
    color=efficiency_color,
    marker=">",
    linestyle="solid",
    markersize=8,
    label=media_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
grey_arrow = mlines.Line2D(
    [],
    [],
    color=satisfaction_color,
    marker=">",
    linestyle="dashed",
    markersize=8,
    label=media_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[blue_arrow, grey_arrow], bbox_to_anchor=(0.45, 0), ncol=2)

blue_arrow = mlines.Line2D(
    [],
    [],
    color=efficiency_color,
    marker=">",
    linestyle="solid",
    markersize=8,
    label=departments_labels_media[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
grey_arrow = mlines.Line2D(
    [],
    [],
    color=satisfaction_color,
    marker=">",
    linestyle="dashed",
    markersize=8,
    label=departments_labels_media[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[blue_arrow, grey_arrow], bbox_to_anchor=(0.85, 0), ncol=2)

# Update labels and titles
ax1.set_xlabel(xlabel)
ax1.set_title(title_media)
ax2.set_title(title_departments_media)
fig.suptitle(suptitle)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("quiver_44.pdf", bbox_inches="tight")
