
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines
import matplotlib.patches as mpatches

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Define the vector field function for financial metrics
def financial_vector_field(X, Y):
    U = -Y * 0.5
    V = X * 1.5
    return U, V

def modified_financial_vector_field(X, Y):
    U = -0.5 - X**2 * 0.3 + Y * 0.4
    V = 0.5 + X * 0.7 - Y**2 * 0.6
    return U, V

# Create a grid of points for financial data
x = np.linspace(1, 50, 10)
y = np.linspace(1, 50, 10)
X, Y = np.meshgrid(x, y)

# Compute the vector field for financial data
U, V = financial_vector_field(X, Y)

# Compute the modified vector field for a different financial scenario
U_mod, V_mod = modified_financial_vector_field(X, Y)

# Plot the curves representing financial growth as inverse functions with slight variations
x = np.linspace(1, 50, 100)
xlabel = "Revenue (in thousands)"
ylabel = "Expenses (in thousands)"
patch_labels = ["Baseline Financial Field", "Modified Financial Field"]
line_labels = ["Q1 Data", "Q2 Data", "Projected Growth", "Forecast"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(10, 8))
color1 = "#1f77b4"
color2 = "#ff7f0e"
color3 = "#2ca02c"
color4 = "#d62728"
color5 = "#9467bd"
color6 = "#8c564b"

# Plot quiver and curves together
plt.quiver(X, Y, U, V, color=color1, alpha=0.6)
plt.quiver(X, Y, U_mod, V_mod, color=color2, alpha=0.6)

plt.plot(x, np.clip(50 / (x**0.5), 0, 50), color=color3, linestyle="-", linewidth=2)
plt.plot(x, np.clip(45 / (x**0.5 + 2), 0, 50), color=color4, linestyle="--", linewidth=2)
plt.plot(x, np.clip(45 / (x**0.7 + 2), 0, 50), color=color5, linestyle="-.", linewidth=2)
plt.plot(x, np.clip(60 / (x**0.7 + 3), 0, 50), color=color6, linestyle=":", linewidth=2)

# Add labels and legend
plt.xlabel(xlabel, fontsize=16, style="italic")
plt.ylabel(ylabel, fontsize=16, style="italic")

red_patch = mpatches.Patch(color=color1, label=patch_labels[0], alpha=0.6)
blue_patch = mpatches.Patch(color=color2, label=patch_labels[1], alpha=0.6)

# Create legend for curves
train_line = mlines.Line2D([], [], color=color3, label=line_labels[0], linestyle="-")
test_line = mlines.Line2D([], [], color=color4, label=line_labels[1], linestyle="--")
sindy_train_line = mlines.Line2D(
    [], [], color=color5, label=line_labels[2], linestyle="-."
)
sindy_test_line = mlines.Line2D(
    [], [], color=color6, label=line_labels[3], linestyle=":"
)

# Combine all legend handles
handles = [
    red_patch,
    blue_patch,
    train_line,
    test_line,
    sindy_train_line,
    sindy_test_line,
]

# Add the legend to the plot with specified location
plt.legend(handles=handles, loc="lower left")

# Set y-axis limit to 100
plt.ylim(0, 50)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("quiver_50.pdf", bbox_inches="tight")
