
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# New dataset for Psychological Analysis focusing on Different Factors
factors = ["Stress", "Happiness", "Performance", "Anxiety", "Motivation", "Satisfaction"]

# Data for the subplots representing two different timelines.
diversity_data = [[2.0, 3.5, 4.1, 3.2, 4.5, 2.8], [2.8, 4.8, 5.5, 3.9, 5.2, 3.5]]
accuracy_data = [[65, 70, 55, 85, 50, 90], [75, 80, 95, 85, 80, 93]]
scatter_sizes = [100, 150, 130, 170, 220, 200]

# Legend labels for the subplots.
ax1_legend_names = ["Short-term", "6 Months", "1 Year", "2 Years"]
ax2_legend_names = ["Short-term", "Mid-term", "Long-term", "5 Years"]
ax1_legend_title = "Assessment Period"
ax2_legend_title = "Evaluation Duration"
xlabel = "Variability in Assessment"
ylabel = "Self-Assessment Scores (%)"
suptitle = "Psychological Factors and Their Assessment Over Time"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create 2x1 grid of subplots with a specified figure size.
fig, axs = plt.subplots(2, 1, figsize=(8, 8))

# Use new colors for the points to distinguish them better.
colors = ["#1f77b4", "#ff7f0e", "#2ca02c", "#d62728", "#9467bd", "#8c564b"]

# Populate the subplots with scatter points and add text labels.
for idx, ax in enumerate(axs):
    for factor, div, acc, size, color in zip(
        factors, diversity_data[idx], accuracy_data[idx], scatter_sizes, colors
    ):
        ax.scatter(
            div, acc, s=size, color=color, alpha=0.6, edgecolors="w", linewidth=1
        )  # Plot the scatter points.
        ax.text(
            div, acc + 2, factor, fontsize=9, ha="center", color="black", weight="bold"
        )  # Add text labels above scatter points.

    ax.set_xlabel(xlabel, fontsize=12)  # X-axis label.
    ax.set_ylabel(ylabel, fontsize=12)  # Y-axis label.
    ax.grid(True, linestyle="--", alpha=0.6)  # Add grid lines for better readability.
    ax.tick_params(
        axis="both", which="major", labelsize=10
    )  # Customize tick parameters.

# Adjust the x and y limits and ticks for the subplots.
axs[0].set_xlim(1.5, 5.0)
axs[0].set_ylim(50, 95)
axs[0].set_xticks(np.arange(1.5, 5.1, 0.5))
axs[0].set_yticks(np.arange(50, 96, 10))
axs[1].set_xlim(2.5, 6.0)
axs[1].set_ylim(70, 100)
axs[1].set_xticks(np.arange(2.5, 6.1, 0.5))
axs[1].set_yticks(np.arange(70, 101, 10))

size_legend_handles = [100, 150, 170, 220]  # Sizes for the legend handles.

# Create custom legend handles for the first subplot.
ax1_legend_handles = [
    mlines.Line2D(
        [],
        [],
        color=color,
        marker="o",
        linestyle="None",
        markersize=(size**0.5) * 0.8,
        label=name,
    )
    for size, name, color in zip(
        size_legend_handles, ax1_legend_names, colors[: len(ax1_legend_names)]
    )
]

# Create custom legend handles for the second subplot.
ax2_legend_handles = [
    mlines.Line2D(
        [],
        [],
        color=color,
        marker="o",
        linestyle="None",
        markersize=(size**0.5) * 0.8,
        label=name,
    )
    for size, name, color in zip(
        size_legend_handles, ax2_legend_names, colors[: len(ax2_legend_names)]
    )
]

# Add legends to the subplots.
axs[0].legend(
    handles=ax1_legend_handles,
    loc="upper left",
    title=ax1_legend_title,
    labelspacing=1.5,
    edgecolor="gray",
)
axs[1].legend(
    handles=ax2_legend_handles,
    loc="upper left",
    title=ax2_legend_title,
    labelspacing=1.5,
    edgecolor="gray",
)

# Add a common title for the entire figure
fig.suptitle(suptitle, fontsize=14, fontweight="bold")

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout(rect=[0, 0, 1, 0.96])
plt.savefig("scatter_33.pdf", bbox_inches="tight")
