

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)  # Set a random seed for reproducibility

# Generate 5 equidistant mean values between 2 and 8 (representing crop yields in tons/ha)
mean_values = np.linspace(2, 8, 5)
# Use standard deviations to reflect realistic agricultural data variability
standard_deviations = [0.8] * 5

data_practice1 = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values, standard_deviations)
]
data_practice2 = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values, standard_deviations)
]
positions_practice1 = np.array(range(1, len(data_practice1) + 1)) - 0.2
positions_practice2 = np.array(range(1, len(data_practice2) + 1)) + 0.2

# Labels and titles
legend_labels = ["Practice 1", "Practice 2"]
xlabel = "Season"
ylabel = "Crop Yield (tons/ha)"
xticks = [1, 2, 3, 4, 5]
xtickslabels = ["Season 1", "Season 2", "Season 3", "Season 4", "Season 5"]


# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis with the specified dimensions
fig, ax = plt.subplots(figsize=(8, 7))

# Narrower violin plots
violin_width = 0.35  # set this to a suitable value

# Create the violin plot with adjusted positions
parts_practice1 = ax.violinplot(
    data_practice1, positions=positions_practice1, widths=violin_width, showmeans=False, showmedians=True
)
parts_practice2 = ax.violinplot(
    data_practice2, positions=positions_practice2, widths=violin_width, showmeans=False, showmedians=True
)

# Customizing the colors of the violin plot
for pc in parts_practice1["bodies"]:
    pc.set_facecolor("#75a3a0")  # Muted teal
    pc.set_edgecolor("black")
    pc.set_alpha(1)

for pc in parts_practice2["bodies"]:
    pc.set_facecolor("#e89c72")  # Muted coral
    pc.set_edgecolor("black")
    pc.set_alpha(1)

# Customizing the median lines and removing caps
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = parts_practice1[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

    vp = parts_practice2[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

# Customizing the colors of the violin plot and adding statistics
for i in range(len(data_practice1)):
    # Adding the statistical annotations for data_practice1
    quartile1, median, quartile3 = np.percentile(data_practice1[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_practice1[i][data_practice1[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_practice1[i][data_practice1[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(positions_practice1[i], quartile1, quartile3, color="black", linestyle="-", lw=4)
    ax.hlines(
        median,
        positions_practice1[i] - 0.025,
        positions_practice1[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_practice1[i], lower_whisker, upper_whisker, color="black", linestyle="-", lw=1
    )

    # Adding the statistical annotations for data_practice2
    quartile1, median, quartile3 = np.percentile(data_practice2[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_practice2[i][data_practice2[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_practice2[i][data_practice2[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(positions_practice2[i], quartile1, quartile3, color="black", linestyle="-", lw=4)
    ax.hlines(
        median,
        positions_practice2[i] - 0.025,
        positions_practice2[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_practice2[i], lower_whisker, upper_whisker, color="black", linestyle="-", lw=1
    )

# Change the border color to grey
for spine in ax.spines.values():
    spine.set_edgecolor("grey")

# Remove small ticks beside the numbers on the x and y axes
ax.tick_params(axis="both", which="both", length=0)

# Adding the corrected legend
ax.legend(
    [parts_practice1["bodies"][0], parts_practice2["bodies"][0]],
    legend_labels,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.1),
    ncol=2,
)

# Setting the title and labels
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Setting the x-axis labels
ax.set_xticks(xticks)
ax.set_xticklabels(xtickslabels)

# Enabling y-axis grid lines
ax.yaxis.grid(
    True, linestyle="-", linewidth=0.7, color="grey", zorder=0
)  # Customize the style of the grid

# ===================
# Part 4: Saving Output
# ===================
# Adjust figure size to match original image's dimensions
fig.set_size_inches(7, 5)  # Width, Height in inches

# Adjust layout for better fit
plt.tight_layout()

# Display the plot
plt.savefig("violin_23.pdf", bbox_inches="tight")
