
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(24)
# Generate new sample data for the domain of environmental awareness
data_environment = np.random.beta(a=[2.5, 3.5, 4.5], b=[4, 3, 2], size=(100, 3))
data_environment_memory = np.random.beta(a=[3.5, 4.5, 5.5], b=[4, 3, 2], size=(100, 3))

categories = ["Energy Conservation", "Recycling Rate", "Water Usage Efficiency"]
violin_width = 0.02

# Axes Limits and Labels
ylabel_value = "Awareness Score"
labels = ["Without Historical Awareness", "With Historical Awareness"]
title = "Comparison of Environmental Awareness Aspects"
supertitle = "Environmental Awareness Distribution"
legend_labels = ["No Historical Influence", "With Historical Influence"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(8, 8))

# Define the categories and the colors for each group
colors = ["#9acd32", "#4682b4"]  # Green for without historical awareness, Blue for with historical awareness

# The scaling factor is used to ensure the violins do not overlap
scaling_factor = 1

# Define offset to separate the half violin plots in the single Axes object
offsets = [-0.05, 0, 0.05]

# Plot the half-violins with an offset
for i, category in enumerate(categories):
    offset = offsets[i]

    # Plot data without historical awareness
    kde_data = gaussian_kde(data_environment[:, i])
    kde_x = np.linspace(0, 1, 300)
    kde_data_y = kde_data(kde_x)
    kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        kde_data_y_scaled * scaling_factor + offset,
        offset,
        color=colors[0],
        edgecolor="#6b8e23",
        alpha=0.6,
    )

    # Plot data with historical awareness
    kde_data_memory = gaussian_kde(data_environment_memory[:, i])
    kde_data_memory_y = kde_data_memory(kde_x)
    kde_data_memory_y_scaled = kde_data_memory_y / max(kde_data_memory_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        offset,
        -kde_data_memory_y_scaled * scaling_factor + offset,
        color=colors[1],
        edgecolor="#4682b4",
        alpha=0.6,
    )
    ax.text(
        offset, -0.1, category, ha="center", va="top"
    )  # Add the category label below the violin plot

# Set x and y axis labels and limits
ax.set_xlim(
    min(offsets) - scaling_factor * violin_width - 0.01,
    max(offsets) + scaling_factor * violin_width + 0.01,
)
y_margin = 0.01  # Adding 5% margin at top and bottom of the y-axis
y_min, y_max = ax.get_ylim()
ax.set_ylim(y_min - y_margin, y_max + y_margin)
ax.set_ylabel(ylabel_value)
ax.set_xticks([])  # Remove x-ticks as they are not meaningful here

# Adjust the legend
handles = [
    plt.Rectangle((0, 0), 1, 1, color=color, edgecolor="#9e8d8b") for color in colors
]
ax.legend(handles, legend_labels, loc="lower left", bbox_to_anchor=(0.6, -0.2), ncol=1)

# Add title and super title
plt.title(title)
plt.suptitle(supertitle)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better fit and save the plot
plt.tight_layout()
plt.savefig("violin_24.pdf", bbox_inches="tight")
