
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Generate synthetic data for historical scholarly traits
data_classical = np.random.beta(a=[1.5, 2.5, 3], b=[2.5, 2, 1.5], size=(100, 3))
data_medieval = np.random.beta(a=[2.5, 1.5, 2], b=[1.5, 3, 2.5], size=(100, 3))

categories = ["Reasoning", "Faith", "Scientific Inquiry"]
violin_width = 0.02

# Axes Limits and Labels
ylabel_value = "Scholarly Score"
xlabel_value = ""
title_value = "Comparison of Historical Scholarly Traits"
legend_labels = ["Classical Era", "Medieval Era"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(6, 6))

# Define the categories and the colors for each group
colors = ["#a6cee3", "#b2df8a"]

# The scaling factor is used to ensure the violins do not overlap
scaling_factor = 1

# Define offset to separate the half violin plots in the single Axes object
offsets = [-0.05, 0, 0.05]

# Plot the half-violins with an offset
for i, category in enumerate(categories):
    offset = offsets[i]

    # Plot data for classical era
    kde_data = gaussian_kde(data_classical[:, i])
    kde_x = np.linspace(0, 1, 300)
    kde_data_y = kde_data(kde_x)
    kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        kde_data_y_scaled * scaling_factor + offset,
        offset,
        color=colors[0],
        edgecolor="#6a9fb5",
    )

    # Plot data for medieval era
    kde_data_memory = gaussian_kde(data_medieval[:, i])
    kde_data_memory_y = kde_data_memory(kde_x)
    kde_data_memory_y_scaled = kde_data_memory_y / max(kde_data_memory_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        offset,
        -kde_data_memory_y_scaled * scaling_factor + offset,
        color=colors[1],
        edgecolor="#86b87d",
    )
    ax.text(offset, -0.1, category, ha="center", va="top")  # Add the category label below the violin plot

# Set x and y axis labels and limits
ax.set_xlim(
    min(offsets) - scaling_factor * violin_width - 0.01,
    max(offsets) + scaling_factor * violin_width + 0.01,
)
y_margin = 0.01  # Adding margin at top and bottom of the y-axis
y_min, y_max = ax.get_ylim()
ax.set_ylim(y_min - y_margin, y_max + y_margin)
ax.set_ylabel(ylabel_value)
ax.set_xticks([])  # Remove x-ticks

# Set title
ax.set_title(title_value, pad=20)

# Adjust the legend
handles = [
    plt.Rectangle((0, 0), 1, 1, color=color, edgecolor="#6a9fb5" if color == colors[0] else "#86b87d") for color in colors
]
ax.legend(handles, legend_labels, loc="lower left", bbox_to_anchor=(0.6, -0.2), ncol=1)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better fit and save the plot
plt.tight_layout()
plt.savefig("violin_25.pdf", bbox_inches="tight")
