
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(42)
# New sample data for Society Metrics
data = np.random.normal(loc=[50, 60, 55], scale=[10, 15, 12], size=(100, 3))
data_memory = np.random.normal(loc=[55, 65, 58], scale=[8, 13, 10], size=(100, 3))

categories = ["Happiness Index", "Social Support", "Education Level"]
violin_width = 0.02

# Axes Limits and Labels
ylabel_value = "Society Metrics Score"
labels = ["Current Data", "Historical Data"]
title = "Societal Metrics Comparison"
supertitle = "Comparison of Key Societal Metrics"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(8, 8))

# Define the categories and the colors for each group
colors = ["#228B22", "#1E90FF"]  # Green, Blue

# The scaling factor is used to ensure the violins do not overlap
scaling_factor = 1

# Define offset to separate the half violin plots in the single Axes object
offsets = [-0.05, 0, 0.05]

# Plot the half-violins with an offset
for i, category in enumerate(categories):
    offset = offsets[i]

    # Plot data without memory
    kde_data = gaussian_kde(data[:, i])
    kde_x = np.linspace(20, 100, 300)
    kde_data_y = kde_data(kde_x)
    kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        kde_data_y_scaled * scaling_factor + offset,
        offset,
        color=colors[0],
        edgecolor="#006400",
    )

    # Plot data with memory
    kde_data_memory = gaussian_kde(data_memory[:, i])
    kde_data_memory_y = kde_data_memory(kde_x)
    kde_data_memory_y_scaled = kde_data_memory_y / max(kde_data_memory_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        offset,
        -kde_data_memory_y_scaled * scaling_factor + offset,
        color=colors[1],
        edgecolor="#00008B",
    )
    ax.text(
        offset, 15, category, ha="center", va="top"
    )  # Add the category label below the violin plot

# Set x and y axis labels and limits
ax.set_xlim(
    min(offsets) - scaling_factor * violin_width - 0.01,
    max(offsets) + scaling_factor * violin_width + 0.01,
)
y_margin = 5  # Adding margin at top and bottom of the y-axis
ax.set_ylim(20 - y_margin, 100 + y_margin)
ax.set_ylabel(ylabel_value)
ax.set_xticks([])  # Remove x-ticks as they are not meaningful here

# Adding titles
ax.set_title(title)
fig.suptitle(supertitle, fontsize=14)

# Adjust the legend
handles = [
    plt.Rectangle((0, 0), 1, 1, color=color, edgecolor=edge) for color, edge in zip(colors, ["#006400", "#00008B"])
]
ax.legend(handles, labels, loc="lower left", bbox_to_anchor=(0.6, -0.2), ncol=1)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better fit and save the plot
plt.tight_layout()
plt.savefig("violin_26.pdf", bbox_inches="tight")
