
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Generate fictitious data representing happiness levels in different age groups
data_4_age_groups = np.random.beta(a=[2, 2, 3, 2], b=[2, 3, 2, 5], size=(100, 4))
data_memory_4_age_groups = np.random.beta(a=[3, 4, 2, 2], b=[2, 1, 2, 4], size=(100, 4))

# Generate fictitious data representing stress levels in different occupational groups
data_2_occupation_groups = np.random.beta(a=[3, 2], b=[2, 3], size=(100, 2))
data_memory_2_occupation_groups = np.random.beta(a=[2, 3], b=[3, 2], size=(100, 2))

ylabel = "Psychological Metric"
ylim = [0, 1]
violin_width = 0.5
scaling_factor = 1
kde_x = np.linspace(0, 1, 300)

# Offsets for groups
offsets_4_age_groups = np.linspace(-3, 3, 4)
offsets_2_occupation_groups = np.linspace(-1, 1, 2)
labels = ["Before Intervention", "After Intervention"]
titles = ["Age Group Happiness Levels", "Occupation Group Stress Levels"]
legend_labels = ["Study A", "Study B"]
xticklabels = [["Child", "Teen", "Adult", "Elderly"], ["White-collar", "Blue-collar"]]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(10, 5))

# Define the colors for each group
colors_4_age_groups = ["#1f77b4", "#ff7f0e"]
colors_2_occupation_groups = ["#1f77b4", "#ff7f0e"]

# Function to plot half violins
def plot_half_violins(
    ax, data, data_memory, offsets, colors, labels, title, xticklabels
):
    # Plot the half-violins with an offset for groups
    for i in range(data.shape[1]):
        offset = offsets[i]

        # Plot data before intervention
        kde_data = gaussian_kde(data[:, i])
        kde_x = np.linspace(0, 1, 300)
        kde_data_y = kde_data(kde_x)
        kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
        ax.fill_betweenx(
            kde_x,
            kde_data_y_scaled * scaling_factor + offset,
            offset,
            color=colors[0],
            edgecolor="#1f77b4",
        )

        # Plot data after intervention
        kde_data_memory = gaussian_kde(data_memory[:, i])
        kde_data_memory_y = kde_data_memory(kde_x)
        kde_data_memory_y_scaled = (
            kde_data_memory_y / max(kde_data_memory_y) * violin_width
        )
        ax.fill_betweenx(
            kde_x,
            offset,
            -kde_data_memory_y_scaled * scaling_factor + offset,
            color=colors[1],
            edgecolor="#ff7f0e",
        )

    # Set x and y axis labels, limits, and add x-axis tick labels for groups
    ax.set_xlim(
        min(offsets) - scaling_factor - violin_width,
        max(offsets) + scaling_factor + violin_width,
    )
    ax.set_ylim(ylim)  # Set y-axis limits to 0-1 for beta distribution
    ax.set_ylabel(ylabel)
    ax.set_xticks(offsets)  # Set x-ticks to the center of each group
    ax.set_xticklabels(xticklabels)  # Label x-ticks accordingly
    ax.title.set_text(title)

# Plot each set of violins
plot_half_violins(
    ax1,
    data_4_age_groups,
    data_memory_4_age_groups,
    offsets_4_age_groups,
    colors_4_age_groups,
    labels,
    titles[0],
    xticklabels[0],
)
plot_half_violins(
    ax2,
    data_2_occupation_groups,
    data_memory_2_occupation_groups,
    offsets_2_occupation_groups,
    colors_2_occupation_groups,
    labels,
    titles[1],
    xticklabels[1],
)

# Add a legend for the entire figure
handles = [
    plt.Line2D(
        [0], [0], marker="o", color="w", markerfacecolor="#1f77b4", markersize=10
    ),
    plt.Line2D(
        [0], [0], marker="o", color="w", markerfacecolor="#ff7f0e", markersize=10
    ),
]

fig.legend(
    handles, legend_labels, loc="lower center", bbox_to_anchor=(0.5, -0.05), ncol=2
)

# ===================
# Part 4: Saving Output
# ===================
# Tighten the layout and show the combined plot
plt.tight_layout()

# Display the plot
plt.savefig("violin_28.pdf", bbox_inches="tight")
