
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate new data for stock market performance
data_4_sectors = np.random.normal(loc=[0.001, 0.002, 0.0015, 0.0025], scale=[0.02, 0.015, 0.025, 0.03], size=(100, 4))
data_memory_4_sectors = np.random.normal(loc=[0.0015, 0.0025, 0.002, 0.003], scale=[0.02, 0.015, 0.025, 0.03], size=(100, 4))
ylabel = "Daily Percentage Change"
ylim = [-0.1, 0.1]
violin_width = 0.5
scaling_factor = 1
kde_x = np.linspace(-0.1, 0.1, 300)

# Offsets for sectors
offsets_4_sectors = np.linspace(-3, 3, 4)
labels = ["Without Strategy", "With Strategy"]
titles = ["Daily Stock Performance", "Strategy Applied Performance"]
legend_labels = ["Current Performance", "Strategy Performance"]
xticklabels = [
    ["Technology", "Healthcare", "Finance", "Energy"],
    ["Technology", "Healthcare", "Finance", "Energy"],
]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(10, 5))

# Define the colors for each sector
colors_4_sectors = ["#66c2a5", "#fc8d62"]

# Function to plot half violins
def plot_half_violins(
    ax, data, data_memory, offsets, colors, labels, title, xticklabels
):
    # Plot the half-violins with an offset for sectors
    for i in range(data.shape[1]):
        offset = offsets[i]

        # Plot data without strategy
        kde_data = gaussian_kde(data[:, i])
        kde_x = np.linspace(-0.1, 0.1, 300)
        kde_data_y = kde_data(kde_x)
        kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
        ax.fill_betweenx(
            kde_x,
            kde_data_y_scaled * scaling_factor + offset,
            offset,
            color=colors[0],
            edgecolor="#ffffff",
        )

        # Plot data with strategy
        kde_data_memory = gaussian_kde(data_memory[:, i])
        kde_data_memory_y = kde_data_memory(kde_x)
        kde_data_memory_y_scaled = (
            kde_data_memory_y / max(kde_data_memory_y) * violin_width
        )
        ax.fill_betweenx(
            kde_x,
            offset,
            -kde_data_memory_y_scaled * scaling_factor + offset,
            color=colors[1],
            edgecolor="#ffffff",
        )

    # Set x and y axis labels, limits, and add x-axis tick labels for sectors
    ax.set_xlim(
        min(offsets) - scaling_factor - violin_width,
        max(offsets) + scaling_factor + violin_width,
    )
    ax.set_ylim(ylim)  # Set y-axis limits for daily percentage change
    ax.set_ylabel(ylabel)
    ax.set_xticks(offsets)  # Set x-ticks to the center of each sector
    ax.set_xticklabels(xticklabels)  # Label x-ticks accordingly
    ax.title.set_text(title)

# Plot each set of violins
plot_half_violins(
    ax1,
    data_4_sectors,
    data_memory_4_sectors,
    offsets_4_sectors,
    colors_4_sectors,
    labels,
    titles[0],
    xticklabels[0],
)
plot_half_violins(
    ax2,
    data_4_sectors,
    data_memory_4_sectors,
    offsets_4_sectors,
    colors_4_sectors,
    labels,
    titles[1],
    xticklabels[1],
)

# Add a legend for the entire figure
handles = [
    plt.Line2D(
        [0], [0], marker="o", color="w", markerfacecolor="#fc8d62", markersize=10
    ),
    plt.Line2D(
        [0], [0], marker="o", color="w", markerfacecolor="#66c2a5", markersize=10
    ),
]

fig.legend(
    handles, legend_labels, loc="lower center", bbox_to_anchor=(0.5, -0.1), ncol=2
)

# ===================
# Part 4: Saving Output
# ===================
# Tighten the layout and show the combined plot
plt.tight_layout()

# Display the plot
plt.savefig("violin_29.pdf", bbox_inches="tight")
