
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde
import random

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Sample data to represent temperature readings at different weather stations
data = np.random.normal(loc=[30, 25, 20, 15, 10], scale=[5, 7, 6, 4, 5], size=(15, 5))
data_memory = np.random.normal(
    loc=[28, 22, 18, 14, 12], scale=[6, 6, 5, 4, 4], size=(45, 5)
)

# Labels and legend
xticklabels = ["Station A", "Station B", "Station C", "Station D", "Station E"]
legend_labels = ["Current Year", "Previous Year"]

# The scaling factor is used to ensure the violins do not overlap
scaling_factor = 1
violin_width = 0.5

# Adjust the offsets for 5 groups instead of 3
offsets = np.linspace(-3, 3, 5)

xlabel = "Weather Stations"
ylabel = "Temperature (°C)"
title = "Distribution of Temperatures Across Weather Stations"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(8, 6))

# Define the colors for each group
colors = ["#2e8b57", "#4682b4"]
legend_colors = ["#4682b4", "#2e8b57"]

# Plot the half-violins with an offset for 5 groups
for i in range(data.shape[1]):
    offset = offsets[i]

    # Plot data for the current year
    kde_data = gaussian_kde(data[:, i])
    kde_x = np.linspace(-10, 50, 300)
    kde_data_y = kde_data(kde_x)
    kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        kde_data_y_scaled * scaling_factor + offset,
        offset,
        color=colors[0],
        edgecolor="black",
    )

    # Plot data for the previous year
    kde_data_memory = gaussian_kde(data_memory[:, i])
    kde_data_memory_y = kde_data_memory(kde_x)
    kde_data_memory_y_scaled = kde_data_memory_y / max(kde_data_memory_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        offset,
        -kde_data_memory_y_scaled * scaling_factor + offset,
        color=colors[1],
        edgecolor="black",
    )

    # add yellow stars at the top of each violin plot to represent peak points
    ax.scatter(
        offset,
        random.uniform(-10, 50),
        marker="*",
        color="yellow",
        s=260,
        zorder=3,
        edgecolor="black",
    )

# Set x and y axis labels, limits, and add x-axis tick labels for 5 groups
ax.set_xlim(
    min(offsets) - scaling_factor - violin_width,
    max(offsets) + scaling_factor + violin_width,
)
ax.set_xticks(offsets)  # Set x-ticks to the center of each group
ax.set_xticklabels(xticklabels)  # Label x-ticks accordingly

ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)

# Adjust the legend
handles = [
    plt.Rectangle((0, 0), 1, 1, color=color, edgecolor="black")
    for color in legend_colors
]
ax.legend(handles, legend_labels, loc="upper left", ncol=1)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("violin_34.pdf", bbox_inches="tight")
