

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Generate synthetic brush stroke intensity data for different art styles
np.random.seed(0)

# Define mean and std dev for different art styles
impressionism_data = np.clip(np.random.normal(70, 10, 200), 40, 100)
surrealism_data = np.clip(np.random.normal(60, 12, 200), 30, 90)
cubism_data = np.clip(np.random.normal(50, 15, 200), 20, 80)
expressionism_data = np.clip(np.random.normal(80, 8, 200), 50, 100)
realism_data = np.clip(np.random.normal(55, 14, 200), 25, 85)

data = [impressionism_data, surrealism_data, cubism_data, expressionism_data, realism_data]
categories = ["Impressionism", "Surrealism", "Cubism", "Expressionism", "Realism"]
ylabel = "Brush Stroke Intensity"
ylim = [10, 110]
xlabel = "Art Styles"
textlabels = ["Mean Intensity", "Intensity Range"]

# Calculate mean and intensity range for annotations
mean_intensity = [np.mean(d) for d in data]
intensity_range = [np.ptp(d) for d in data]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 7))

# Create violin plots
violin_parts = ax.violinplot(data, showmeans=False, showmedians=True, showextrema=False)

# Customize the appearance
ax.set_ylabel(ylabel)
ax.set_xticks(np.arange(1, len(categories) + 1))
ax.set_xticklabels(categories)
ax.set_ylim(ylim)
ax.set_xlabel(xlabel)

# Customize colors and annotations
colors = ["#FF5733", "#FF8D1A", "#FFC300", "#DAF7A6", "#AED6F1"]
for i, (pc, d) in enumerate(zip(violin_parts["bodies"], data)):
    pc.set_facecolor(colors[i])
    pc.set_edgecolor("black")
    pc.set_alpha(1)

    # Calculate quartiles and median
    quartile1, median, quartile3 = np.percentile(d, [25, 50, 75])
    iqr = quartile3 - quartile1

    # Calculate whiskers
    lower_whisker = np.min(d[d >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(d[d <= quartile3 + 1.5 * iqr])

    # Annotate statistics
    ax.vlines(i + 1, quartile1, quartile3, color="k", linestyle="-", lw=4)
    ax.scatter(i + 1, median, color="w", s=10, zorder=3)
    ax.vlines(i + 1, lower_whisker, upper_whisker, color="k", linestyle="-", lw=1)
    ax.text(i + 1 + 0.3, median, f"{median:.2f}", ha="left", va="center", color="black", rotation=45)

    # Annotate with Mean Intensity and Intensity Range
    ax.text(i + 1, 18, f"{mean_intensity[i]:.1f}", ha="center", va="center", color="green", fontsize=10)
    ax.text(i + 1, 13, f"{intensity_range[i]:.1f}", ha="center", va="center", color="blue", fontsize=10)

ax.text(5.6, 18, textlabels[0], ha="left", va="center", color="green", fontsize=10)
ax.text(5.6, 13, textlabels[1], ha="left", va="center", color="blue", fontsize=10)

# Make other parts of the violin plots invisible
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = violin_parts.get(partname)
    if vp:
        vp.set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better fit
plt.tight_layout()

# Display the plot
plt.savefig("violin_42.pdf", bbox_inches="tight")

