

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Sample data to mimic historical figures' popularity scores
data = np.random.normal(loc=[0.7, 0.5, 0.6], scale=[0.12, 0.1, 0.15], size=(100, 3))
data_memory = np.random.normal(loc=[0.65, 0.55, 0.62], scale=[0.1, 0.14, 0.13], size=(100, 3))
kde_x = np.linspace(0, 1, 300)

# Define the categories and the colors for each group
categories = ["Ancient Times", "Medieval Era", "Modern Era"]
title = "Comparison of Historical Figures' Popularity Scores Over Time"
ylabel = "Popularity Score"
labels = ["Regular Data", "Adjusted Data"]

# Define offset to separate the half violin plots in the single Axes object
offsets = [-0.05, 0, 0.05]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(6, 6))
colors = ["#2ca02c", "#d62728"]  # Green and Red for data representation
# The scaling factor is used to ensure the violins do not overlap
scaling_factor = 1
violin_width = 0.02

# Plot the half-violins with an offset
for i, category in enumerate(categories):
    offset = offsets[i]
    # Plot data without adjustment
    kde_data = gaussian_kde(data[:, i])
    kde_data_y = kde_data(kde_x)
    kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        kde_data_y_scaled * scaling_factor + offset,
        offset,
        color=colors[0],
        edgecolor="black",
        label=labels[0] if i == 0 else "",
    )

    # Plot data with adjustment
    kde_data_memory = gaussian_kde(data_memory[:, i])
    kde_data_memory_y = kde_data_memory(kde_x)
    kde_data_memory_y_scaled = kde_data_memory_y / max(kde_data_memory_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        offset,
        -kde_data_memory_y_scaled * scaling_factor + offset,
        color=colors[1],
        edgecolor="black",
        label=labels[1] if i == 0 else "",
    )

    # Plot the mean as a star marker for data without adjustment
    ax.plot(
        offset,
        np.mean(data[:, i]),
        "*",
        color="white",
        markersize=12,
        markeredgecolor="black",
    )
    # Plot the mean as a star marker for data with adjustment
    ax.plot(
        offset,
        np.mean(data_memory[:, i]),
        "*",
        color="white",
        markersize=12,
        markeredgecolor="black",
    )

    # Add the category label below the violin plot
    ax.text(
        offset, -0.1, category, ha="center", va="top", fontsize=9
    )

# Add title
ax.set_title(title)

# Add grid
ax.grid(True)

# Set x and y axis labels and limits
ax.set_xlim(
    min(offsets) - scaling_factor * violin_width - 0.06,
    max(offsets) + scaling_factor * violin_width + 0.06,
)
y_margin = 0.02  # Adding margin at top and bottom of the y-axis
y_min, y_max = ax.get_ylim()
ax.set_ylim(y_min - y_margin, y_max + y_margin)
ax.set_ylabel(ylabel)
ax.set_xticks([])  # Remove x-ticks as they are not meaningful here

# Adjust the legend
handles = [
    plt.Rectangle((0, 0), 1, 1, color=color, edgecolor="black") for color in colors
]

ax.legend(handles, labels, loc="lower center", bbox_to_anchor=(0.5, -0.15), ncol=2)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better fit and save the figure
plt.tight_layout()
plt.savefig("violin_48.pdf", bbox_inches="tight")

