

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(42)

# Sample data to mimic tourism ratings
data = np.random.beta(a=[2, 5, 3], b=[3, 2, 4], size=(100, 3))
data_memory = np.random.beta(a=[3, 4, 4], b=[4, 3, 2], size=(100, 3))
kde_x = np.linspace(0, 1, 300)

# Define the categories and the colors for each group
categories = ["Scenery", "Accessibility", "Amenities"]
title = "Comparison of Tourism Ratings with and without Guided Tours"
# Define offset to separate the half violin plots in the single Axes object
offsets = [-0.05, 0, 0.05]
ylabel = "Rating"
labels = ["Without Guided Tour", "With Guided Tour"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(6, 6))
colors = ["#8e44ad", "#3498db"]  # Using a different color palette

# The scaling factor is used to ensure the violins do not overlap
scaling_factor = 1
violin_width = 0.02

# Plot the half-violins with an offset
for i, category in enumerate(categories):
    offset = offsets[i]
    # Plot data without guided tour
    kde_data = gaussian_kde(data[:, i])
    kde_data_y = kde_data(kde_x)
    kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        kde_data_y_scaled * scaling_factor + offset,
        offset,
        color=colors[0],
        edgecolor="black",
        alpha=0.6,
    )

    # Plot data with guided tour
    kde_data_memory = gaussian_kde(data_memory[:, i])
    kde_data_memory_y = kde_data_memory(kde_x)
    kde_data_memory_y_scaled = kde_data_memory_y / max(kde_data_memory_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        offset,
        -kde_data_memory_y_scaled * scaling_factor + offset,
        color=colors[1],
        edgecolor="black",
        alpha=0.6,
    )

    # Plot the mean as a diamond marker for data without guided tour
    ax.plot(
        offset,
        np.mean(data[:, i]),
        "d",
        color="white",
        markersize=10,
        markeredgecolor="black",
    )
    # Plot the mean as a diamond marker for data with guided tour
    ax.plot(
        offset,
        np.mean(data_memory[:, i]),
        "d",
        color="white",
        markersize=10,
        markeredgecolor="black",
    )

    ax.text(offset, -0.1, category, ha="center", va="top", fontsize=9)  # Add the category label below the violin plot

# Add title
ax.set_title(title, fontsize=14, weight='bold')

# Add grid
ax.grid(True, linestyle='--', alpha=0.7)

# Set x and y axis labels and limits
ax.set_xlim(
    min(offsets) - scaling_factor * violin_width - 0.06,
    max(offsets) + scaling_factor * violin_width + 0.06,
)
y_margin = 0.02  # Adding margin at top and bottom of the y-axis
y_min, y_max = ax.get_ylim()
ax.set_ylim(y_min - y_margin, y_max + y_margin)
ax.set_ylabel(ylabel, fontsize=12)
ax.set_xticks([])  # Remove x-ticks as they are not meaningful here

# Adjust the legend
handles = [
    plt.Rectangle((0, 0), 1, 1, color=color, edgecolor="black") for color in colors
]

ax.legend(handles, labels, loc="lower center", bbox_to_anchor=(0.5, -0.15), ncol=2, fontsize=10)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better fit and save the figure
plt.tight_layout()
plt.savefig("violin_50.pdf", bbox_inches="tight")

