
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
# Setting a random seed for reproducibility
np.random.seed(42)

# Generate mean stock prices for 5 months for 2 sectors
mean_prices_tech = np.linspace(150, 200, 5)
mean_prices_health = np.linspace(100, 130, 5)
# Smaller standard deviations for realistic price clustering
standard_deviations = [15] * 5

# Generate data for Tech and Health sectors
tech_prices = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_prices_tech, standard_deviations)
]
health_prices = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_prices_health, standard_deviations)
]
positions_tech = np.array(range(1, len(tech_prices) + 1)) - 0.2
positions_health = np.array(range(1, len(health_prices) + 1)) + 0.2

legend_labels = ["Tech Sector", "Health Sector"]
xlabel = "Month"
ylabel = "Stock Prices ($)"
xticks = [1, 2, 3, 4, 5]
xtickslabels = ["Jan", "Feb", "Mar", "Apr", "May"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis with specified dimensions
fig, ax = plt.subplots(figsize=(8, 7))

# Violin plot width
violin_width = 0.35

# Create the violin plot with adjusted positions
parts_tech = ax.violinplot(
    tech_prices,
    positions=positions_tech,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)
parts_health = ax.violinplot(
    health_prices,
    positions=positions_health,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)

# Customizing the colors of the violin plot
for pc in parts_tech["bodies"]:
    pc.set_facecolor("royalblue")  # Tech Sector color
    pc.set_edgecolor("black")
    pc.set_alpha(0.7)

for pc in parts_health["bodies"]:
    pc.set_facecolor("mediumseagreen")  # Health Sector color
    pc.set_edgecolor("black")
    pc.set_alpha(0.7)

# Customizing the median lines and removing caps
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = parts_tech[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

    vp = parts_health[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

# Adding statistics annotations for both sectors
for i in range(len(tech_prices)):
    # Tech Sector statistics
    quartile1, median, quartile3 = np.percentile(tech_prices[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(tech_prices[i][tech_prices[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(tech_prices[i][tech_prices[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_tech[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_tech[i] - 0.025,
        positions_tech[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_tech[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

    # Health Sector statistics
    quartile1, median, quartile3 = np.percentile(health_prices[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(health_prices[i][health_prices[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(health_prices[i][health_prices[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_health[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_health[i] - 0.025,
        positions_health[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_health[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

# Change the border color to grey
for spine in ax.spines.values():
    spine.set_edgecolor("grey")

# Remove small ticks beside the numbers on the x and y axes
ax.tick_params(axis="both", which="both", length=0)

# Adding the corrected legend
ax.legend(
    [parts_tech["bodies"][0], parts_health["bodies"][0]],
    legend_labels,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.1),
    ncol=2,
)

# Setting the title and labels
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Setting the x-axis labels
ax.set_xticks(xticks)
ax.set_xticklabels(xtickslabels)

# Enabling y-axis grid lines
ax.yaxis.grid(True, linestyle="-", linewidth=0.7, color="grey", zorder=0)

# ===================
# Part 4: Saving Output
# ===================
# Adjust figure size to match original image's dimensions
fig.set_size_inches(7, 5)  # Width, Height in inches

# Adjust layout for better fit
plt.tight_layout()

# Display the plot
plt.savefig("violin_54.pdf", bbox_inches="tight")
