

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Setting a random seed for reproducibility
import numpy as np

np.random.seed(0)

# Generate mean messages sent for 5 months for 2 platforms
mean_messages_platform1 = np.linspace(200, 500, 5)
mean_messages_platform2 = np.linspace(250, 450, 5)
standard_deviations = [50] * 5

# Generate data for Platform1 and Platform2
platform1_messages = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_messages_platform1, standard_deviations)
]
platform2_messages = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_messages_platform2, standard_deviations)
]
positions_platform1 = np.array(range(1, len(platform1_messages) + 1)) - 0.2
positions_platform2 = np.array(range(1, len(platform2_messages) + 1)) + 0.2
legend_labels = ["Platform1", "Platform2"]
xlabel = "Month"
ylabel = "Average Messages Sent"
xticks = [1, 2, 3, 4, 5]
xtickslabels = ["Jan", "Feb", "Mar", "Apr", "May"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis with specified dimensions
fig, ax = plt.subplots(figsize=(8, 7))

# Violin plot width
violin_width = 0.35

# Create the violin plot with adjusted positions
parts_platform1 = ax.violinplot(
    platform1_messages,
    positions=positions_platform1,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)
parts_platform2 = ax.violinplot(
    platform2_messages,
    positions=positions_platform2,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)

# Customizing the colors of the violin plot
for pc in parts_platform1["bodies"]:
    pc.set_facecolor("deepskyblue")  # Platform1 color
    pc.set_edgecolor("black")
    pc.set_alpha(0.7)

for pc in parts_platform2["bodies"]:
    pc.set_facecolor("lightseagreen")  # Platform2 color
    pc.set_edgecolor("black")
    pc.set_alpha(0.7)

# Customizing the median lines and removing caps
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = parts_platform1[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

    vp = parts_platform2[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

# Adding statistics annotations for both platforms
for i in range(len(platform1_messages)):
    # Platform1 statistics
    quartile1, median, quartile3 = np.percentile(platform1_messages[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(platform1_messages[i][platform1_messages[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(platform1_messages[i][platform1_messages[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_platform1[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_platform1[i] - 0.025,
        positions_platform1[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_platform1[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

    # Platform2 statistics
    quartile1, median, quartile3 = np.percentile(platform2_messages[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(platform2_messages[i][platform2_messages[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(platform2_messages[i][platform2_messages[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_platform2[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_platform2[i] - 0.025,
        positions_platform2[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_platform2[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

# Change the border color to grey
for spine in ax.spines.values():
    spine.set_edgecolor("grey")

# Remove small ticks beside the numbers on the x and y axes
ax.tick_params(axis="both", which="both", length=0)

# Adding the corrected legend
ax.legend(
    [parts_platform1["bodies"][0], parts_platform2["bodies"][0]],
    legend_labels,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.1),
    ncol=2,
)

# Setting the title and labels
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Setting the x-axis labels
ax.set_xticks(xticks)
ax.set_xticklabels(xtickslabels)

# Enabling y-axis grid lines
ax.yaxis.grid(True, linestyle="-", linewidth=0.7, color="grey", zorder=0)

# ===================
# Part 4: Saving Output
# ===================
# Adjust figure size to match original image's dimensions
fig.set_size_inches(7, 5)  # Width, Height in inches

# Adjust layout for better fit
plt.tight_layout()

# Display the plot
plt.savefig("violin_55.pdf", bbox_inches="tight")
