
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Setting a random seed for reproducibility
import numpy as np

np.random.seed(0)
# Generate mean monthly revenues for 2 streaming platforms over six months
mean_revenues_platform1 = np.linspace(400, 600, 6)
mean_revenues_platform2 = np.linspace(350, 500, 6)
# Smaller standard deviations for realistic revenue clustering
standard_deviations = [40] * 6

# Generate data for Platform1 and Platform2
platform1_revenues = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_revenues_platform1, standard_deviations)
]
platform2_revenues = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_revenues_platform2, standard_deviations)
]
positions_platform1 = np.array(range(1, len(platform1_revenues) + 1)) - 0.2
positions_platform2 = np.array(range(1, len(platform2_revenues) + 1)) + 0.2

# Labels and title
legend_labels = ["Platform1", "Platform2"]
xlabel = "Month"
ylabel = "Revenue ($ in millions)"
xticks = [1, 2, 3, 4, 5, 6]
xtickslabels = ["Jan", "Feb", "Mar", "Apr", "May", "Jun"]
title = "Monthly Subscription Revenue Comparison"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis with specified dimensions
fig, ax = plt.subplots(figsize=(8, 7))

# Violin plot width
violin_width = 0.35

# Create the violin plot with adjusted positions
parts_platform1 = ax.violinplot(
    platform1_revenues,
    positions=positions_platform1,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)
parts_platform2 = ax.violinplot(
    platform2_revenues,
    positions=positions_platform2,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)

# Customizing the colors of the violin plot
for pc in parts_platform1["bodies"]:
    pc.set_facecolor("forestgreen")  # Platform1 color
    pc.set_edgecolor("black")
    pc.set_alpha(0.7)

for pc in parts_platform2["bodies"]:
    pc.set_facecolor("royalblue")  # Platform2 color
    pc.set_edgecolor("black")
    pc.set_alpha(0.7)

# Customizing the median lines and removing caps
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = parts_platform1[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

    vp = parts_platform2[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

# Adding statistics annotations for both platforms
for i in range(len(platform1_revenues)):
    # Platform1 statistics
    quartile1, median, quartile3 = np.percentile(platform1_revenues[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(platform1_revenues[i][platform1_revenues[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(platform1_revenues[i][platform1_revenues[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_platform1[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_platform1[i] - 0.025,
        positions_platform1[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_platform1[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

    # Platform2 statistics
    quartile1, median, quartile3 = np.percentile(platform2_revenues[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(platform2_revenues[i][platform2_revenues[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(platform2_revenues[i][platform2_revenues[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_platform2[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_platform2[i] - 0.025,
        positions_platform2[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_platform2[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

# Change the border color to grey
for spine in ax.spines.values():
    spine.set_edgecolor("grey")

# Remove small ticks beside the numbers on the x and y axes
ax.tick_params(axis="both", which="both", length=0)

# Adding the corrected legend
ax.legend(
    [parts_platform1["bodies"][0], parts_platform2["bodies"][0]],
    legend_labels,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.0),
    ncol=2,
)

# Setting the title and labels
ax.set_title(title)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Setting the x-axis labels
ax.set_xticks(xticks)
ax.set_xticklabels(xtickslabels)

# Enabling y-axis grid lines
ax.yaxis.grid(True, linestyle="-", linewidth=0.7, color="grey", zorder=0)

# ===================
# Part 4: Saving Output
# ===================
# Adjust figure size to match original image's dimensions
fig.set_size_inches(7, 5)  # Width, Height in inches

# Adjust layout for better fit
plt.tight_layout()

# Display the plot
plt.savefig("violin_56.pdf", bbox_inches="tight")
