

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate mean values for drawing times for two types of artists
mean_values_artist_a = np.linspace(5, 10, 5)  # Experienced Artist
mean_values_artist_b = np.linspace(15, 30, 5)  # Novice Artist

# Standard deviations for project times
standard_deviations = [3] * 5

# Generate synthetic data for Artist A
data_artist_a = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values_artist_a, standard_deviations)
]

# Generate synthetic data for Artist B
data_artist_b = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values_artist_b, standard_deviations)
]

# Set positions for violins
positions_artist_a = np.array(range(1, len(data_artist_a) + 1)) - 0.2
positions_artist_b = np.array(range(1, len(data_artist_b) + 1)) + 0.2

# Labels and ticks
legend_labels = ["Experienced Artist", "Novice Artist"]
xlabel = "Project Types"
ylabel = "Drawing Time (hours)"
xticks = [1, 2, 3, 4, 5]
xtickslabels = ["Sketch", "Portrait", "Landscape", "Abstract", "Mural"]
# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(8, 7))

violin_width = 0.35

# Create violin plots
parts_artist_a = ax.violinplot(
    data_artist_a,
    positions=positions_artist_a,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)
parts_artist_b = ax.violinplot(
    data_artist_b,
    positions=positions_artist_b,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)

# Customizing colors
for pc in parts_artist_a["bodies"]:
    pc.set_facecolor("lightgreen")
    pc.set_edgecolor("black")
    pc.set_alpha(0.7)

for pc in parts_artist_b["bodies"]:
    pc.set_facecolor("salmon")
    pc.set_edgecolor("black")
    pc.set_alpha(0.7)

# Customizing median lines and removing caps
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp_artist_a = parts_artist_a[partname]
    vp_artist_b = parts_artist_b[partname]
    vp_artist_a.set_edgecolor("black")
    vp_artist_b.set_edgecolor("black")
    vp_artist_a.set_linewidth(1)
    vp_artist_b.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp_artist_a.set_visible(False)
        vp_artist_b.set_visible(False)

# Adding statistical annotations
for i in range(len(data_artist_a)):
    quartile1, median, quartile3 = np.percentile(data_artist_a[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_artist_a[i][data_artist_a[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_artist_a[i][data_artist_a[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_artist_a[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_artist_a[i] - 0.025,
        positions_artist_a[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_artist_a[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

    quartile1, median, quartile3 = np.percentile(data_artist_b[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_artist_b[i][data_artist_b[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_artist_b[i][data_artist_b[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_artist_b[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_artist_b[i] - 0.025,
        positions_artist_b[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_artist_b[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

# Customize borders
for spine in ax.spines.values():
    spine.set_edgecolor("grey")

# Remove ticks
ax.tick_params(axis="both", which="both", length=0)

# Adding legend
ax.legend(
    [parts_artist_a["bodies"][0], parts_artist_b["bodies"][0]],
    legend_labels,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.1),
    ncol=2,
)

# Setting labels
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Setting x-axis labels
ax.set_xticks(xticks)
ax.set_xticklabels(xtickslabels)

# Enabling y-axis grid lines
ax.yaxis.grid(True, linestyle="-", linewidth=0.7, color="grey", zorder=0)

# ===================
# Part 4: Saving Output
# ===================
fig.set_size_inches(7, 5)

plt.tight_layout()
plt.savefig("violin_57.pdf", bbox_inches="tight")

