

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Generate mean values for training hours for Running and Cycling
mean_values_running = np.linspace(1, 2.5, 5)
mean_values_cycling = np.linspace(2, 3.5, 5)

# Standard deviations
standard_deviations_running = [0.5] * 5
standard_deviations_cycling = [0.7] * 5

# Generate synthetic data for Running
data_running = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values_running, standard_deviations_running)
]

# Generate synthetic data for Cycling
data_cycling = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values_cycling, standard_deviations_cycling)
]

# Set positions for violins
positions_running = np.array(range(1, len(data_running) + 1)) - 0.2
positions_cycling = np.array(range(1, len(data_cycling) + 1)) + 0.2

# Labels and ticks
legend_labels = ["Running", "Cycling"]
xlabel = "Weekdays"
ylabel = "Training Hours"
xticks = [1, 2, 3, 4, 5]
xtickslabels = ["Mon", "Tue", "Wed", "Thu", "Fri"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(8, 7))

violin_width = 0.35

# Create violin plots
parts_running = ax.violinplot(
    data_running,
    positions=positions_running,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)
parts_cycling = ax.violinplot(
    data_cycling,
    positions=positions_cycling,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)

# Customizing colors
for pc in parts_running["bodies"]:
    pc.set_facecolor("lightgreen")
    pc.set_edgecolor("black")
    pc.set_alpha(0.8)

for pc in parts_cycling["bodies"]:
    pc.set_facecolor("lightcoral")
    pc.set_edgecolor("black")
    pc.set_alpha(0.8)

# Customizing median lines and removing caps
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp_running = parts_running[partname]
    vp_cycling = parts_cycling[partname]
    vp_running.set_edgecolor("black")
    vp_cycling.set_edgecolor("black")
    vp_running.set_linewidth(1)
    vp_cycling.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp_running.set_visible(False)  # Hide caps
        vp_cycling.set_visible(False)  # Hide caps

# Adding statistical annotations
for i in range(len(data_running)):
    quartile1, median, quartile3 = np.percentile(data_running[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_running[i][data_running[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_running[i][data_running[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_running[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_running[i] - 0.025,
        positions_running[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_running[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

    quartile1, median, quartile3 = np.percentile(data_cycling[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_cycling[i][data_cycling[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_cycling[i][data_cycling[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_cycling[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_cycling[i] - 0.025,
        positions_cycling[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_cycling[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

# Customize borders
for spine in ax.spines.values():
    spine.set_edgecolor("grey")

# Remove ticks
ax.tick_params(axis="both", which="both", length=0)

# Adding legend
ax.legend(
    [parts_running["bodies"][0], parts_cycling["bodies"][0]],
    legend_labels,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.1),
    ncol=2,
)

# Setting labels
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Setting x-axis labels
ax.set_xticks(xticks)
ax.set_xticklabels(xtickslabels)

# Enabling y-axis grid lines
ax.yaxis.grid(True, linestyle="-", linewidth=0.7, color="grey", zorder=0)

# ===================
# Part 4: Saving Output
# ===================
fig.set_size_inches(7, 5)

plt.tight_layout()
plt.savefig("violin_59.pdf", bbox_inches="tight")
