
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
import random

# Generate new data for energy consumption of smart home devices
np.random.seed(0)
data = np.random.beta(a=[2, 3, 5, 4, 2], b=[3, 2, 4, 3, 2], size=(10, 5))
data_memory = np.random.beta(a=[4, 3, 3, 2, 5], b=[3, 4, 2, 3, 4], size=(40, 5))

xticklabels = ["Smart Thermostat", "Smart Light", "Smart Speaker", "Smart Lock", "Smart Plug"]
legend_labels = ["Current Data", "Historical Data"]
scaling_factor = 1
violin_width = 0.5
xlabel = "Device Type"
ylabel = "Energy Consumption (kWh)"
title = "Energy Consumption of Smart Home Devices"

# Adjust the offsets for 5 groups
offsets = np.linspace(-3, 3, 5)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(10, 7))  # Increased size for better spacing

# Define the colors for each group
colors = ["#76c7c0", "#8c564b"]  # Teal and Brown for smart home
legend_colors = ["#76c7c0", "#8c564b"]

# Plot the half-violins with an offset for 5 groups
for i in range(data.shape[1]):
    offset = offsets[i]

    # Plot data without memory
    kde_data = gaussian_kde(data[:, i])
    kde_x = np.linspace(0, 1, 300)
    kde_data_y = kde_data(kde_x)
    kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        kde_data_y_scaled * scaling_factor + offset,
        offset,
        color=colors[0],
        edgecolor="black",
        alpha=0.7,
    )

    # Plot data with memory
    kde_data_memory = gaussian_kde(data_memory[:, i])
    kde_data_memory_y = kde_data_memory(kde_x)
    kde_data_memory_y_scaled = kde_data_memory_y / max(kde_data_memory_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        offset,
        -kde_data_memory_y_scaled * scaling_factor + offset,
        color=colors[1],
        edgecolor="black",
        alpha=0.7,
    )

    # Add stars at the top of each violin plot for emphasis
    ax.scatter(
        offset,
        random.uniform(0.2, 0.8),
        marker="*",
        color="#fdd835",  # Yellow star
        s=260,
        zorder=3,
        edgecolor="black",
    )

# Set axis labels and limits
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_xlim(
    min(offsets) - scaling_factor - violin_width,
    max(offsets) + scaling_factor + violin_width,
)
ax.set_xticks(offsets)
ax.set_xticklabels(xticklabels)

# Adding grid lines for better readability
ax.grid(True, linestyle="--", alpha=0.5)

# Set title
plt.title(title)

# Adjust the legend
handles = [
    plt.Rectangle((0, 0), 1, 1, color=color, edgecolor="black")
    for color in legend_colors
]
ax.legend(handles, legend_labels, loc="upper left", ncol=1)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("violin_74.pdf", bbox_inches="tight")
