
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Generate plausible data for temperature variation in different regions
data = np.random.normal(loc=[12, 15, 10, 20, 8], scale=[5, 4, 6, 3, 5], size=(10, 5))
data_memory = np.random.normal(loc=[11, 14, 9, 18, 7], scale=[6, 5, 7, 4, 6], size=(40, 5))

xticklabels = [
    "North America",
    "South America",
    "Europe",
    "Asia",
    "Australia",
]
legend_labels = ["Current Year", "Historical Average"]
scaling_factor = 1
violin_width = 0.5

# Adjust the offsets for 5 groups
offsets = np.linspace(-3, 3, 5)

xlabel = "Geographical Regions"
ylabel = "Temperature Distribution"
title = "Temperature Variation Across Different Regions"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(10, 8))  # Increased size for better visualization

# Define the colors for each group
colors = ["#87CEEB", "#FFB6C1"]  # Soft sky blue and light pink
legend_colors = ["#FFB6C1", "#87CEEB"]

# Plot the half-violins with an offset for 5 groups
for i in range(data.shape[1]):
    offset = offsets[i]

    # Plot data without memory
    kde_data = gaussian_kde(data[:, i])
    kde_x = np.linspace(min(data[:, i])-5, max(data[:, i])+5, 300)
    kde_data_y = kde_data(kde_x)
    kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        kde_data_y_scaled * scaling_factor + offset,
        offset,
        facecolor=colors[0],
        edgecolor="black",
        alpha=0.6,
    )

    # Plot data with memory
    kde_data_memory = gaussian_kde(data_memory[:, i])
    kde_data_memory_y = kde_data_memory(kde_x)
    kde_data_memory_y_scaled = kde_data_memory_y / max(kde_data_memory_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        offset,
        -kde_data_memory_y_scaled * scaling_factor + offset,
        facecolor=colors[1],
        edgecolor="black",
        alpha=0.6,
    )

    # Add stylish markers at the top of each violin plot
    ax.scatter(
        offset,
        np.mean(kde_x),
        marker="D",
        color="purple",
        s=100,
        zorder=3,
        edgecolor="black",
    )

# Set x and y axis labels, limits, and add x-axis tick labels for 5 groups
ax.set_xlim(
    min(offsets) - scaling_factor - violin_width,
    max(offsets) + scaling_factor + violin_width,
)
ax.set_xticks(offsets)
ax.set_xticklabels(xticklabels, fontsize=12, fontweight="bold")
ax.set_xlabel(xlabel, fontsize=14, fontweight="bold")
ax.set_ylabel(ylabel, fontsize=14, fontweight="bold")
ax.set_title(title, fontsize=16, fontweight="bold")

# Adding grid lines for better readability
ax.grid(True, linestyle="--", alpha=0.5)

# Adjust the legend
handles = [
    plt.Line2D(
        [0],
        [0],
        color=color,
        lw=10,
        linestyle="-",
        marker="D",
        markersize=15,
        markerfacecolor=color,
        markeredgewidth=3,
    )
    for color in legend_colors
]
ax.legend(
    handles,
    legend_labels,
    loc="upper left",
    fontsize=12,
    title="Legend",
    title_fontsize="13",
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("violin_77.pdf", bbox_inches="tight")
