# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.collections import PolyCollection

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
# Set a random seed for reproducibility
np.random.seed(0)
# Function to create polygon under graph
def polygon_under_graph(x, y):
    return [(x[0], 0.0), *zip(x, y), (x[-1], 0.0)]

# Data for bar chart
years = [2000, 2001, 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009]
diagnosis = [2.1, 2.3, 2.5, 2.8, 3.0, 3.2, 3.5, 3.8, 4.0, 4.2]
treatment = [1.5, 1.6, 1.7, 2.0, 2.1, 2.3, 2.5, 2.7, 2.8, 3.0]

# Data for distribution graph
x = np.linspace(0.0, 10.0, 31)
technological_advances = range(1, 4)
exp = np.exp
verts = [
    polygon_under_graph(x, exp(-0.5 * (x - 2 * t) ** 2)) for t in technological_advances
]  # Gaussian distributions

# Labels and titles
xlabel_year = "Year"
ylabel_medical_field = "Medical Field"
zlabel_investment = "Investment (Million USD)"
yticks_labels = ["Diagnosis", "Treatment"]
title_bar_chart = "Investment in Medical Research Fields"
xlabel_time = "Time Since Innovation (Years)"
ylabel_tech_advancement = "Technological Advancement"
zlabel_adoption_rate = "Adoption Rate"
title_distribution_graph = "Adoption Rate of Medical Technologies"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
facecolors = plt.get_cmap("viridis")(np.linspace(0, 1, len(verts)))
# Initialize figure and axes
fig = plt.figure(figsize=(15, 10))
ax1 = fig.add_subplot(121, projection="3d")  # 3D bar chart
ax2 = fig.add_subplot(122, projection="3d")  # 3D distribution graph

# Plot data for bar chart
ax1.bar(years, diagnosis, zs=0, zdir="y", color="#28a745", alpha=0.8)
ax1.bar(years, treatment, zs=1, zdir="y", color="#0077b6", alpha=0.8)

# Set labels and ticks for bar chart
ax1.set_xlabel(xlabel_year)
ax1.set_ylabel(ylabel_medical_field)
ax1.set_zlabel(zlabel_investment)
ax1.set_yticks([0, 1])
ax1.set_yticklabels(yticks_labels)
ax1.set_title(title_bar_chart, pad=20)

# Add polygons to the distribution graph
poly = PolyCollection(verts, facecolors=facecolors, alpha=0.7)
ax2.add_collection3d(poly, zs=technological_advances, zdir="y")

# Set labels and limits for distribution graph
ax2.set(
    xlim=(0, 10),
    ylim=(1, 4),
    zlim=(0, 1),
    xlabel=xlabel_time,
    ylabel=ylabel_tech_advancement,
    zlabel=zlabel_adoption_rate,
)
ax2.set_yticks([1, 2, 3])
ax2.set_title(title_distribution_graph, pad=20)

# ===================
# Part 4: Saving Output
# ===================
# Adjust the layout and save the figure
plt.tight_layout()
plt.savefig("3d_19.pdf", bbox_inches="tight")