import matplotlib.pyplot as plt
import numpy as np; np.random.seed(0); np.random.seed(0)

from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
# Generating sample data representing traffic trends for three different decades
traffic_90s = np.random.normal(50, 5, 100)  # Traffic in the 1990s
traffic_00s = np.random.normal(60, 7, 100)  # Traffic in the 2000s
traffic_10s = np.random.normal(70, 9, 100)  # Traffic in the 2010s

# Creating KDE for each sample data set to estimate the density of data points
kde_90s = gaussian_kde(traffic_90s)
kde_00s = gaussian_kde(traffic_00s)
kde_10s = gaussian_kde(traffic_10s)

traffic_range = np.linspace(30, 90, 50)  # Defining the range of traffic volume for plotting
labels = ["1990s", "2000s", "2010s"]
ax1_legend_title = "Decade"
ax1_xlabel = "Density"
ax2_ylabel = "Traffic Volume"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Setting up the figure and axes for a 1 x 2 layout
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(10, 5))

# First subplot: Fill between plot along y-axis (KDE Plot)
ax1.fill_betweenx(traffic_range, kde_90s(traffic_range), color="skyblue", alpha=0.4)
ax1.fill_betweenx(traffic_range, kde_00s(traffic_range), color="sandybrown", alpha=0.5)
ax1.fill_betweenx(traffic_range, kde_10s(traffic_range), color="olivedrab", alpha=0.3)
ax1.plot(kde_90s(traffic_range), traffic_range, label=labels[0], color="blue")
ax1.plot(kde_00s(traffic_range), traffic_range, label=labels[1], color="orange")
ax1.plot(kde_10s(traffic_range), traffic_range, label=labels[2], color="green")
ax1.legend(title=ax1_legend_title, loc="upper right")
ax1.set_xlabel(ax1_xlabel)
ax1.set_yticks([])  # Hiding y-axis ticks for clarity

# Second subplot: Box plot for the same datasets along y-axis
box = ax2.boxplot(
    [traffic_90s, traffic_00s, traffic_10s], vert=True, patch_artist=True, medianprops={"color": "black"}
)
colors = ["skyblue", "sandybrown", "olivedrab"]  # Color matching with KDE plot
for patch, color in zip(box["boxes"], colors):
    patch.set_facecolor(color)

ax2.set_ylabel(ax2_ylabel)
ax2.set_xticks([])  # Hiding x-axis ticks for clarity
ax2.set_ylim(30, 90)  # Setting limits for y-axis to align with the KDE plot

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig('CB_28.pdf', bbox_inches='tight')