# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np; np.random.seed(0)
# Updated gradient steps
months = np.linspace(0, 100, 50)

# Generate different trends for each line
ev_adoption_rate = np.cos(months * 0.1) + 1.2  # Cosine trend (EV adoption rate)
battery_efficiency = np.log(months + 1) * 0.3 + 0.5  # Logarithmic growth (battery efficiency improvements)
daily_production_variability = np.random.normal(
    loc=1.0, scale=0.2, size=len(months)
)  # Random noise (daily production variability)
renewable_energy_share = 1.0 / (1.0 + np.exp(-0.05 * (months - 50)))  # Sigmoid growth (renewable energy share)

# Simulate standard deviations for error
error_margin = 0.15
ev_std_dev = np.full_like(ev_adoption_rate, error_margin)
battery_std_dev = np.full_like(battery_efficiency, error_margin)
production_std_dev = np.full_like(daily_production_variability, error_margin)
energy_std_dev = np.full_like(renewable_energy_share, error_margin)

# Labels and configuration
x_axis_label = "Months"
y_axis_label = "Sustainability Metrics"
plot_labels = ["Daily Production Variability", "Renewable Energy Share"]
x_ticks_main = np.linspace(0, 100, 9)
y_ticks_main = np.arange(0, 3, 0.5)
inset_axes_position = [0.6, 0.15, 0.3, 0.2]
inset_x_limits = [75, 100]
inset_y_limits = [0.5, 1.5]
inset_x_ticks = [75, 85, 95]
inset_y_ticks = [0.5, 1.0, 1.5]
arrow_start = (60, 1.8)
arrow_end = (0.4, 0.6)
zoom_rect_x = [75, 100]
zoom_rect_y = [0.5, 1.5]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with a single plot
fig, ax = plt.subplots(figsize=(6, 6))

# Plot the third line on the main plot
ax.plot(
    months, daily_production_variability, "*--", color="yellow", label=plot_labels[0]
)
ax.fill_between(
    months,
    daily_production_variability - production_std_dev,
    daily_production_variability + production_std_dev,
    color="blue",
    alpha=0.2,
)

# Plot the fourth line on the main plot
ax.plot(
    months, renewable_energy_share, "^-", color="green", label=plot_labels[1]
)
ax.fill_between(
    months,
    renewable_energy_share - energy_std_dev,
    renewable_energy_share + energy_std_dev,
    color="orange",
    alpha=0.2,
)

# Set labels, ticks, legend and grid for the main plot
ax.set_xlabel(x_axis_label, fontsize=12)
ax.set_ylabel(y_axis_label, fontsize=12)
ax.set_xticks(x_ticks_main)
ax.set_yticks(y_ticks_main)
ax.legend(loc="upper left", shadow=True, frameon=True, framealpha=0.9)
ax.grid(
    True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5
)
ax.set_facecolor("#f9f9f9")

# Draw a rectangle on the main plot to indicate the area of zoom-in
ax.plot([zoom_rect_x[0], zoom_rect_x[1]], [zoom_rect_y[1], zoom_rect_y[1]], color="black", lw=1)
ax.plot([zoom_rect_x[0], zoom_rect_x[1]], [zoom_rect_y[0], zoom_rect_y[0]], color="black", lw=1)
ax.plot([zoom_rect_x[0], zoom_rect_x[0]], [zoom_rect_y[0], zoom_rect_y[1]], color="black", lw=1)
ax.plot([zoom_rect_x[1], zoom_rect_x[1]], [zoom_rect_y[0], zoom_rect_y[1]], color="black", lw=1)

# Create the inset with the zoomed-in view
ax_inset = fig.add_axes(
    inset_axes_position
)  # Adjust the position to align with the right side of the main plot

# Plot the third line on the inset
ax_inset.plot(
    months, daily_production_variability, "*--", color="yellow", label=plot_labels[0]
)
ax_inset.fill_between(
    months,
    daily_production_variability - production_std_dev,
    daily_production_variability + production_std_dev,
    color="blue",
    alpha=0.2,
)

# Plot the fourth line on the inset
ax_inset.plot(
    months, renewable_energy_share, "^-", color="green", label=plot_labels[1]
)
ax_inset.fill_between(
    months,
    renewable_energy_share - energy_std_dev,
    renewable_energy_share + energy_std_dev,
    color="orange",
    alpha=0.2,
)

# Set limits, ticks and border color for the inset
ax_inset.set_xlim(inset_x_limits)
ax_inset.set_ylim(inset_y_limits)
ax_inset.set_xticks(inset_x_ticks)
ax_inset.set_yticks(inset_y_ticks)
ax_inset.spines["bottom"].set_color("black")
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

# Add an arrow from the rectangle on the main plot to the inset
ax.annotate(
    "",
    xy=arrow_start,  # Arrow start point (on the main plot)
    xytext=arrow_end,  # Arrow end point (on the inset)
    textcoords="axes fraction",
    arrowprops=dict(facecolor="black", lw=0.1),
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and display the plot
plt.tight_layout()
plt.savefig("PIP_11.pdf", bbox_inches="tight")