import matplotlib.pyplot as plt
import numpy as np; np.random.seed(0); np.random.seed(0)

import matplotlib.gridspec as gridspec

# ===================
# Part 2: Data Preparation
# ===================
# Data for Telemedicine
telemedicine_data = {
    "Benefits": [15.8, 25.4, 30.5, 20.3, 8.0],
    "Reliability": [12.0, 22.5, 35.7, 21.8, 8.0],
    "Security": [10.5, 18.5, 38.0, 24.0, 9.0],
    "Privacy": [20.0, 28.5, 25.5, 18.0, 8.0],
}

# Data for Remote Monitoring
remote_monitoring_data = {
    "Benefits": [20.0, 24.0, 32.0, 14.0, 10.0],
    "Reliability": [10.0, 24.0, 40.0, 20.0, 6.0],
    "Security": [14.0, 18.0, 36.0, 26.0, 6.0],
    "Privacy": [10.0, 22.0, 34.0, 16.0, 18.0],
}

categories = ["Strong Disagree", "Disagree", "Neutral", "Agree", "Strong Agree"]
categories2 = ["Very Low", "Low", "Neutral", "High", "Very High"]
colors = ["#d95f02", "#fdae61", "white", "#a6d96a", "#1a9641"]

labels = ["Telemedicine", "Remote Monitoring"]
xlim = [-70, 70]
xticks = np.arange(-70, 71, 10)


# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a gridspec
gs = gridspec.GridSpec(
    3, 2, height_ratios=[1.2, 0.2, 0.2]
)  # Adjust the height ratios for each row

fig = plt.figure(figsize=(12, 6))

# Create axes using the gridspec
axes = [plt.subplot(gs[0, 0]), plt.subplot(gs[0, 1])]


def create_bar_chart(ax, results, category_names, title):

    labels = list(results.keys())
    data = np.array(list(results.values()))
    data_cum = data.cumsum(axis=1)
    middle_index = data.shape[1] // 2
    offsets = data[:, range(middle_index)].sum(axis=1) + data[:, middle_index] / 2

    # Color Mapping
    category_colors = plt.get_cmap("coolwarm_r")(np.linspace(0.15, 0.85, data.shape[1]))

    # Plot Bars
    for i, (colname, color) in enumerate(zip(category_names, category_colors)):
        widths = data[:, i]
        starts = data_cum[:, i] - widths - offsets
        rects = ax.barh(
            labels,
            widths,
            left=starts,
            height=0.5,
            label=colname,
            color=color,
            edgecolor="black",
        )
        for j, (start, width) in enumerate(zip(starts, widths)):
            # Calculate the center position of each bar segment for the text
            text_x = start + width / 2
            text_y = j  # y-coordinate is based on the bar's index (j)
            ax.text(
                text_x,
                text_y,
                f"{abs(width):.1f}%",
                va="center",
                ha="center",
                color="black",
                fontsize=8,
            )
    # Add Zero Reference Line
    ax.axvline(0, linestyle="-", color="black", alpha=0.25)
    # X Axis
    ax.set_xlim(xlim)
    ax.set_xticks(xticks)
    ax.xaxis.set_major_formatter(lambda x, pos: str(abs(int(x))))
    # Y Axis
    ax.invert_yaxis()
    ax.set_title(title)


# Create bar charts for Eye movement and Brainwave
create_bar_chart(axes[0], telemedicine_data, categories, labels[0])
create_bar_chart(axes[1], remote_monitoring_data, categories2, labels[1])


# Add legend
handles, labels = axes[0].get_legend_handles_labels()
fig.legend(
    handles,
    labels,
    loc="lower center",
    ncol=len(categories),
    bbox_to_anchor=(0.5, 0.25),
)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig('bar_21.pdf', bbox_inches='tight')
